const express = require("express");
const ItemFacturaController = require("../controllers/itemfactura.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     ItemFactura:
 *       type: object
 *       required:
 *         - Cantidad
 *         - CodigoTratamiento
 *         - TarifaTratamiento
 *         - SubTotalTarifa
 *         - CopagoFijo
 *         - TotalCoPagoFijo
 *         - SubTotal
 *         - FacturaId
 *         - TarifarioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del ItemFactura
 *         Cantidad:
 *           type: number
 *           description: Cantidad del Item Factura
 *         CodigoTratamiento:
 *           type: string
 *           description: Codigo Tratamiento del Item Factura
 *         TarifaTratamiento:
 *           type: number
 *           description: Tarifa Tratamiento del Item Factura
 *         SubTotalTarifa:
 *           type: number
 *           description: Sub Total Tarifa del Item Factura
 *         CopagoFijo:
 *           type: number
 *           description: Copago Fijo del Item Factura
 *         TotalCoPagoFijo:
 *           type: number
 *           description: Total CoPago Fijo del Item Factura
 *         SubTotal:
 *           type: number
 *           description: Sub Total del Item Factura
 *         FacturaId:
 *           type: number
 *           description: Identifiador de la Factura
 *         TarifarioId:
 *           type: string
 *           description: Identifiador del Tarifario
 *       example:
 *         id: "1"
 *         Cantidad: "2"
 *         CodigoTratamiento: "4TRE4"
 *         TarifaTratamiento: "120"
 *         SubTotalTarifa: "240"
 *         CopagoFijo: "5"
 *         TotalCoPagoFijo: "10"
 *         SubTotal: "230"
 *         FacturaId: "1"
 *         TarifarioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: ItemFactura
 *   description: API para gestionar ItemFactura
 */
/**
 * @swagger
 * /economia/itemfactura:
 *   get:
 *     summary: Obtiene la lista de ItemFactura
 *     tags: [ItemFactura]
 *     responses:
 *       200:
 *         description: Lista de ItemFactura.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ItemFactura'
 */
router.get("/", verifyToken, ItemFacturaController.getItemFacturaAll);
/**
 * @swagger
 * /economia/itemfactura/{id}:
 *   get:
 *     summary: Obtiene un ItemFactura por ID
 *     tags: [ItemFactura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemFactura
 *     responses:
 *       200:
 *         description: ItemFactura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemFactura'
 *       404:
 *         description: ItemFactura no encontrado.
 */
router.get("/:id", verifyToken, ItemFacturaController.getItemFacturaById);
/**
 * @swagger
 * /economia/itemfactura/create:
 *   post:
 *     summary: Crea un nuevo ItemFactura
 *     tags: [ItemFactura]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ItemFactura'
 *     responses:
 *       201:
 *         description: ItemFactura creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemFactura'
 */
router.post("/create", verifyToken, ItemFacturaController.createItemFactura);
/**
 * @swagger
 * /economia/itemfactura/update/{id}:
 *   put:
 *     summary: Actualiza un ItemFactura existente
 *     tags: [ItemFactura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemFactura a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ItemFactura'
 *     responses:
 *       200:
 *         description: ItemFactura actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemFactura'
 *       404:
 *         description: ItemFactura no encontrado.
 */
router.put("/update/:id", verifyToken, ItemFacturaController.updateItemFactura);
/**
 * @swagger
 * /economia/itemfactura/{id}:
 *   delete:
 *     summary: Elimina un ItemFactura por ID
 *     tags: [ItemFactura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemFactura a eliminar
 *     responses:
 *       200:
 *         description: ItemFactura eliminado exitosamente.
 *       404:
 *         description: ItemFactura no encontrado.
 */
router.delete("/:id", verifyToken, ItemFacturaController.deleteItemFactura);
/**
 * @swagger
 * /economia/itemfactura/filtrar/factura/{factura}:
 *   get:
 *     summary: Obtiene un ItemFactura por factura
 *     tags: [ItemFactura]
 *     parameters:
 *       - in: path
 *         name: factura
 *         schema:
 *           type: number
 *         required: true
 *         description: factura del ItemFactura
 *     responses:
 *       200:
 *         description: ItemFactura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemFactura'
 *       404:
 *         description: ItemFactura no encontrado.
 */
router.get("/filtrar/factura/:factura", verifyToken, ItemFacturaController.getItemFacturaByFactura);
/**
 * @swagger
 * /economia/itemfactura/filtrar/tarifario/{tarifario}:
 *   get:
 *     summary: Obtiene un ItemFactura por tarifario
 *     tags: [ItemFactura]
 *     parameters:
 *       - in: path
 *         name: tarifario
 *         schema:
 *           type: number
 *         required: true
 *         description: tarifario del ItemFactura
 *     responses:
 *       200:
 *         description: ItemFactura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemFactura'
 *       404:
 *         description: ItemFactura no encontrado.
 */
router.get("/filtrar/tarifario/:tarifario", verifyToken, ItemFacturaController.getItemFacturaByTarifario);

module.exports = router;
