const express = require("express");
const ItemRadiografiaController = require("../controllers/itemradiografia.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     ItemRadiografia:
 *       type: object
 *       required:
 *         - Nombre
 *         - RadiografiaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del ItemRadiografia
 *         Nombre:
 *           type: string
 *           description: Nombre del ItemRadiografia
 *         RadiografiaId:
 *           type: string
 *           description: Relacion del ItemRadiografia con la Radiografia
 *       example:
 *         id: "1"
 *         Nombre: "Odontopediatría"
 *         RadiografiaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: ItemRadiografia
 *   description: API para gestionar ItemRadiografia
 */
/**
 * @swagger
 * /radiografia/item:
 *   get:
 *     summary: Obtiene la lista de ItemRadiografia
 *     tags: [ItemRadiografia]
 *     responses:
 *       200:
 *         description: Lista de ItemRadiografia.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ItemRadiografia'
 */
router.get("/", verifyToken, ItemRadiografiaController.getItemRadiografiaAll);
/**
 * @swagger
 * /radiografia/item/{id}:
 *   get:
 *     summary: Obtiene un ItemRadiografia por ID
 *     tags: [ItemRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemRadiografia
 *     responses:
 *       200:
 *         description: ItemRadiografia encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemRadiografia'
 *       404:
 *         description: ItemRadiografia no encontrado.
 */
router.get("/:id", verifyToken, ItemRadiografiaController.getItemRadiografiaById);
/**
 * @swagger
 * /radiografia/item/create:
 *   post:
 *     summary: Crea un nuevo ItemRadiografia
 *     tags: [ItemRadiografia]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ItemRadiografia'
 *     responses:
 *       201:
 *         description: ItemRadiografia creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemRadiografia'
 */
router.post("/create", verifyToken, ItemRadiografiaController.createItemRadiografia);
/**
 * @swagger
 * /radiografia/item/update/{id}:
 *   put:
 *     summary: Actualiza un ItemRadiografia existente
 *     tags: [ItemRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemRadiografia a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ItemRadiografia'
 *     responses:
 *       200:
 *         description: ItemRadiografia actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemRadiografia'
 *       404:
 *         description: ItemRadiografia no encontrado.
 */
router.put("/update/:id", verifyToken, ItemRadiografiaController.updateItemRadiografia);
/**
 * @swagger
 * /radiografia/item/{id}:
 *   delete:
 *     summary: Elimina un ItemRadiografia por ID
 *     tags: [ItemRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemRadiografia a eliminar
 *     responses:
 *       200:
 *         description: ItemRadiografia eliminado exitosamente.
 *       404:
 *         description: ItemRadiografia no encontrado.
 */
router.delete("/:id", verifyToken, ItemRadiografiaController.deleteItemRadiografia);
/**
 * @swagger
 * /radiografia/item/filtro/radiografia/{radiografia}:
 *   get:
 *     summary: Obtiene un ItemRadiografia por radiografia
 *     tags: [ItemRadiografia]
 *     parameters:
 *       - in: path
 *         name: radiografia
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del radiografia
 *     responses:
 *       200:
 *         description: ItemRadiografiaes del radiografia encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemRadiografia'
 *       404:
 *         description: ItemRadiografia no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/radiografia/:radiografia", verifyToken, ItemRadiografiaController.getItemRadiografiaByRadiografia);

module.exports = router;
