const express = require("express");
const ItemRecetaController = require("../controllers/itemreceta.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     ItemReceta:
 *       type: object
 *       required:
 *         - Nombre
 *         - Cantidad
 *         - Tiempo
 *         - Indicacion
 *         - RecetaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Item Receta
 *         Nombre:
 *           type: string
 *           description: Nombre del Item de la Receta
 *         Cantidad:
 *           type: number
 *           description: Cantidad del Item de la Receta
 *         Tiempo:
 *           type: string
 *           description: Tiempo del Item de la Receta
 *         Indicacion:
 *           type: string
 *           description: Indicacion del Item de la Receta
 *         RecetaId:
 *           type: number
 *           description: Identificador del Receta
 *       example:
 *         id: "1"
 *         Nombre: "fadsfsdf"
 *         Cantidad: "12"
 *         Tiempo: "7 dias"
 *         Indicacion: "fasdgsadgdsg"
 *         RecetaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: ItemReceta
 *   description: API para gestionar ItemReceta
 */
/**
 * @swagger
 * /paciente/itemreceta:
 *   get:
 *     summary: Obtiene la lista de ItemReceta
 *     tags: [ItemReceta]
 *     responses:
 *       200:
 *         description: Lista de ItemReceta.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ItemReceta'
 */
router.get("/", verifyToken, ItemRecetaController.getItemRecetaAll);
/**
 * @swagger
 * /paciente/itemreceta/{id}:
 *   get:
 *     summary: Obtiene un ItemReceta por ID
 *     tags: [ItemReceta]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemReceta
 *     responses:
 *       200:
 *         description: ItemReceta encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemReceta'
 *       404:
 *         description: ItemReceta no encontrado.
 */
router.get("/:id", verifyToken, ItemRecetaController.getItemRecetaById);
/**
 * @swagger
 * /paciente/itemreceta/create:
 *   post:
 *     summary: Crea un nuevo ItemReceta
 *     tags: [ItemReceta]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ItemReceta'
 *     responses:
 *       201:
 *         description: ItemReceta creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemReceta'
 */
router.post("/create", verifyToken, ItemRecetaController.createItemReceta); // Aplicamos validación
/**
 * @swagger
 * /paciente/itemreceta/update/{id}:
 *   put:
 *     summary: Actualiza un ItemReceta existente
 *     tags: [ItemReceta]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemReceta a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ItemReceta'
 *     responses:
 *       200:
 *         description: ItemReceta actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemReceta'
 *       404:
 *         description: ItemReceta no encontrado.
 */
router.put("/update/:id", verifyToken, ItemRecetaController.updateItemReceta);
/**
 * @swagger
 * /paciente/itemreceta/{id}:
 *   delete:
 *     summary: Elimina un ItemReceta por ID
 *     tags: [ItemReceta]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ItemReceta a eliminar
 *     responses:
 *       200:
 *         description: ItemReceta eliminado exitosamente.
 *       404:
 *         description: ItemReceta no encontrado.
 */
router.delete("/:id", verifyToken, ItemRecetaController.deleteItemReceta);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/itemreceta/filtrar/receta/{receta}:
 *   get:
 *     summary: Obtiene un ItemReceta por receta
 *     tags: [ItemReceta]
 *     parameters:
 *       - in: path
 *         name: receta
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del receta
 *     responses:
 *       200:
 *         description: ItemReceta encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ItemReceta'
 *       404:
 *         description: ItemReceta no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/receta/:receta", verifyToken, ItemRecetaController.getItemRecetaByReceta);

module.exports = router;