const express = require("express");
const { verifyToken } = require('../middlewares/auth.middleware');
const kardexController = require("../controllers/kardex.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Kardex:
 *       type: object
 *       required:
 *         - Cantidad
 *         - UbicacionInicial
 *         - UbicacionFinal
 *         - Estado
 *         - InsumoId
 *         - AlmacenId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Kardex
 *         Cantidad:
 *           type: number
 *           description: Cantidad a mover
 *         UbicacionInicial:
 *           type: string
 *           description: Ubicacion Inicial del Insumo
 *         UbicacionFinal:
 *           type: string
 *           description: Ubicacion Final del Insumo
 *         Estado:
 *           type: string
 *           description: Estado o motivo de movida (abastecimiento, uso, derivacion, etc)
 *         InsumoId:
 *           type: number
 *           description: Insumo que se esta moviendo
 *         AlmacenId:
 *           type: number
 *           description: SubAlmacenAfectado
 *       example:
 *         id: "1"
 *         Cantidad: "2"
 *         UbicacionInicial: "Almacen Principal"
 *         UbicacionFinal: "Consultorio OdontoPediatra"
 *         Estado: "Abastecimiento"
 *         InsumoId: "1"
 *         AlmacenId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Kardex
 *   description: API para gestionar Kardexs
 */
/**
 * @swagger
 * /inventario/kardex:
 *   get:
 *     summary: Obtiene la lista de Kardexs
 *     tags: [Kardex]
 *     responses:
 *       200:
 *         description: Lista de Kardexs.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Kardex'
 */
router.get("/", verifyToken, kardexController.getKardexAll);
/**
 * @swagger
 * /inventario/kardex/almacen/{almacen}:
 *   get:
 *     summary: Obtiene un Kardex por almacen
 *     tags: [Kardex]
 *     parameters:
 *       - in: path
 *         name: almacen
 *         schema:
 *           type: number
 *         required: true
 *         description: almacen del Kardex
 *     responses:
 *       200:
 *         description: Movimientos en el Almacen
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Kardex'
 *       404:
 *         description: Kardex no encontrado.
 */
router.get("/almacen/:almacen", verifyToken, kardexController.getKardexByAlmacen);
/**
 * @swagger
 * /inventario/kardex/insumo/{insumo}:
 *   get:
 *     summary: Obtiene un Kardex por ID
 *     tags: [Kardex]
 *     parameters:
 *       - in: path
 *         name: insumo
 *         schema:
 *           type: number
 *         required: true
 *         description: insumo del Kardex
 *     responses:
 *       200:
 *         description: Movimientos de tal Isumo de manera global
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Kardex'
 *       404:
 *         description: Kardex no encontrado.
 */
router.get("/insumo/:insumo", verifyToken, kardexController.getKardexByInsumo);
/**
 * @swagger
 * /inventario/kardex/{id}:
 *   get:
 *     summary: Obtiene movimientos de tal insumo en tal almacen
 *     tags: [Kardex]
 *     parameters:
 *       - in: path
 *         name: almacen
 *         schema:
 *           type: number
 *         required: true
 *         description: almacen del Kardex
 *       - in: path
 *         name: insumo
 *         schema:
 *           type: number
 *         required: true
 *         description: insumo del Kardex
 *     responses:
 *       200:
 *         description: Movimientos de tal insumo dentro del almacen
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Kardex'
 *       404:
 *         description: Kardex no encontrado.
 */
router.get("/movimiento/:almacen/:insumo", verifyToken, kardexController.getKardexByMovimientoInsumo);
/**
 * @swagger
 * /inventario/kardex/{id}:
 *   get:
 *     summary: Obtiene un Kardex por ID
 *     tags: [Kardex]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Kardex
 *     responses:
 *       200:
 *         description: Kardex encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Kardex'
 *       404:
 *         description: Kardex no encontrado.
 */
router.get("/:id", verifyToken, kardexController.getKardexById);
/**
 * @swagger
 * /inventario/kardex/create:
 *   post:
 *     summary: Crea un nuevo Kardex
 *     tags: [Kardex]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Kardex'
 *     responses:
 *       201:
 *         description: Kardex creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Kardex'
 */
router.post("/create", verifyToken, kardexController.createKardex);
/**
 * @swagger
 * /inventario/kardex/update/{id}:
 *   put:
 *     summary: Actualiza un Kardex existente
 *     tags: [Kardex]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Kardex a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Kardex'
 *     responses:
 *       200:
 *         description: Kardex actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Kardex'
 *       404:
 *         description: Kardex no encontrado.
 */
router.put("/update/:id", verifyToken, kardexController.updateKardex);
/**
 * @swagger
 * /inventario/kardex/{id}:
 *   delete:
 *     summary: Elimina un Kardex por ID
 *     tags: [Kardex]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Kardex a eliminar
 *     responses:
 *       200:
 *         description: Kardex eliminado exitosamente.
 *       404:
 *         description: Kardex no encontrado.
 */
router.delete("/:id", verifyToken, kardexController.deleteKardex);

// Rutas de filtrado individual


module.exports = router;
