const express = require("express");
const { verifyToken } = require('../middlewares/auth.middleware');
const MovimientoController = require("../controllers/movimiento.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Movimiento:
 *       type: object
 *       required:
 *         - Cantidad
 *         - UbicacionInicial
 *         - UbicacionFinal
 *         - Estado
 *         - InsumoId
 *         - AlmacenId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Movimiento
 *         Cantidad:
 *           type: number
 *           description: Cantidad a mover
 *         UbicacionInicial:
 *           type: string
 *           description: Ubicacion Inicial del Insumo
 *         UbicacionFinal:
 *           type: string
 *           description: Ubicacion Final del Insumo
 *         Estado:
 *           type: string
 *           description: Estado o motivo de movida (abastecimiento, uso, derivacion, etc)
 *         InsumoId:
 *           type: number
 *           description: Insumo que se esta moviendo
 *         AlmacenId:
 *           type: number
 *           description: SubAlmacenAfectado
 *       example:
 *         id: "1"
 *         Cantidad: "2"
 *         UbicacionInicial: "Almacen Principal"
 *         UbicacionFinal: "Consultorio OdontoPediatra"
 *         Estado: "Abastecimiento"
 *         InsumoId: "1"
 *         AlmacenId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Movimiento
 *   description: API para gestionar Movimientos
 */
/**
 * @swagger
 * /inventario/Movimiento:
 *   get:
 *     summary: Obtiene la lista de Movimientos
 *     tags: [Movimiento]
 *     responses:
 *       200:
 *         description: Lista de Movimientos.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Movimiento'
 */
router.get("/", verifyToken, MovimientoController.getMovimientoAll);
/**
 * @swagger
 * /inventario/Movimiento/sede/{sede}:
 *   get:
 *     summary: Obtiene un Movimiento por almacen
 *     tags: [Movimiento]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: identificador de la sede
 *     responses:
 *       200:
 *         description: Movimientos en el Almacen
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Movimiento'
 *       404:
 *         description: Movimiento no encontrado.
 */
router.get("/sede/:sede", verifyToken, MovimientoController.getMovimientoBySede);
/**
 * @swagger
 * /inventario/Movimiento/patrimonio/{patrimonio}:
 *   get:
 *     summary: Obtiene un Movimiento por ID
 *     tags: [Movimiento]
 *     parameters:
 *       - in: path
 *         name: patrimonio
 *         schema:
 *           type: number
 *         required: true
 *         description: identificador del patrimonio
 *     responses:
 *       200:
 *         description: Movimientos de tal Isumo de manera global
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Movimiento'
 *       404:
 *         description: Movimiento no encontrado.
 */
router.get("/patrimonio/:patrimonio", verifyToken, MovimientoController.getMovimientoByPatrimonio);
/**
 * @swagger
 * /inventario/Movimiento/{id}:
 *   get:
 *     summary: Obtiene un Movimiento por ID
 *     tags: [Movimiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Movimiento
 *     responses:
 *       200:
 *         description: Movimiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Movimiento'
 *       404:
 *         description: Movimiento no encontrado.
 */
router.get("/:id", verifyToken, MovimientoController.getMovimientoById);
/**
 * @swagger
 * /inventario/Movimiento/create:
 *   post:
 *     summary: Crea un nuevo Movimiento
 *     tags: [Movimiento]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Movimiento'
 *     responses:
 *       201:
 *         description: Movimiento creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Movimiento'
 */
router.post("/create", verifyToken, MovimientoController.createMovimiento);
/**
 * @swagger
 * /inventario/Movimiento/update/{id}:
 *   put:
 *     summary: Actualiza un Movimiento existente
 *     tags: [Movimiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Movimiento a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Movimiento'
 *     responses:
 *       200:
 *         description: Movimiento actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Movimiento'
 *       404:
 *         description: Movimiento no encontrado.
 */
router.put("/update/:id", verifyToken, MovimientoController.updateMovimiento);
/**
 * @swagger
 * /inventario/Movimiento/{id}:
 *   delete:
 *     summary: Elimina un Movimiento por ID
 *     tags: [Movimiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Movimiento a eliminar
 *     responses:
 *       200:
 *         description: Movimiento eliminado exitosamente.
 *       404:
 *         description: Movimiento no encontrado.
 */
router.delete("/:id", verifyToken, MovimientoController.deleteMovimiento);

// Rutas de filtrado individual


module.exports = router;
