const express = require("express");
const OdontogramaClinicoController = require("../controllers/odontogramaclinico.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     OdontogramaClinico:
 *       type: object
 *       required:
 *         - FechaCreacion
 *         - FechayHoraFin
 *         - Especificaciones
 *         - Observaciones
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del OdontogramaClinico
 *         FechaCreacion:
 *           type: Date
 *           description: Fecha de Creacion del Odontograma Clinico
 *         Especificaciones:
 *           type: string
 *           description: Especificaciones del Odontograma Clinico
 *         Observaciones:
 *           type: string
 *           description: Observaciones del Odontograma Clinico
 *         PacienteId:
 *           type: number
 *           description: Identificador de la Historia Clinica
 *       example:
 *         id: "1"
 *         FechaCreacion: "2025-11-11"
 *         Especificaciones: "fadfasfds fdsaas fdasfasd"
 *         Observaciones: "gadsgadsg dasgsdsd"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: OdontogramaClinico
 *   description: API para gestionar OdontogramaClinico
 */
/**
 * @swagger
 * /odontograma/odontogramaclinico:
 *   get:
 *     summary: Obtiene la lista de OdontogramaClinico
 *     tags: [OdontogramaClinico]
 *     responses:
 *       200:
 *         description: Lista de OdontogramaClinico.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/OdontogramaClinico'
 */
router.get("/", verifyToken, OdontogramaClinicoController.getOdontogramaClinicoAll);
/**
 * @swagger
 * /odontograma/odontogramaclinico/{id}:
 *   get:
 *     summary: Obtiene un OdontogramaClinico por ID
 *     tags: [OdontogramaClinico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OdontogramaClinico
 *     responses:
 *       200:
 *         description: OdontogramaClinico encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OdontogramaClinico'
 *       404:
 *         description: OdontogramaClinico no encontrado.
 */
router.get("/:id", verifyToken, OdontogramaClinicoController.getOdontogramaClinicoById);
/**
 * @swagger
 * /odontograma/odontogramaclinico/create:
 *   post:
 *     summary: Crea un nuevo OdontogramaClinico
 *     tags: [OdontogramaClinico]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OdontogramaClinico'
 *     responses:
 *       201:
 *         description: OdontogramaClinico creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OdontogramaClinico'
 */
router.post("/create", verifyToken, OdontogramaClinicoController.createOdontogramaClinico); // Aplicamos validación
/**
 * @swagger
 * /odontograma/odontogramaclinico/update/{id}:
 *   put:
 *     summary: Actualiza un OdontogramaClinico existente
 *     tags: [OdontogramaClinico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OdontogramaClinico a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OdontogramaClinico'
 *     responses:
 *       200:
 *         description: OdontogramaClinico actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OdontogramaClinico'
 *       404:
 *         description: OdontogramaClinico no encontrado.
 */
router.put("/update/:id", verifyToken, OdontogramaClinicoController.updateOdontogramaClinico);
/**
 * @swagger
 * /odontograma/odontogramaclinico/{id}:
 *   delete:
 *     summary: Elimina un OdontogramaClinico por ID
 *     tags: [OdontogramaClinico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OdontogramaClinico a eliminar
 *     responses:
 *       200:
 *         description: OdontogramaClinico eliminado exitosamente.
 *       404:
 *         description: OdontogramaClinico no encontrado.
 */
router.delete("/:id", verifyToken, OdontogramaClinicoController.deleteOdontogramaClinico);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /odontograma/odontogramaclinico/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un OdontogramaClinico por paciente
 *     tags: [OdontogramaClinico]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: OdontogramaClinico encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OdontogramaClinico'
 *       404:
 *         description: OdontogramaClinico no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, OdontogramaClinicoController.getOdontogramaClinicoByPaciente);

module.exports = router;