const express = require("express");
const OdontogramaEconomicoController = require("../controllers/odontogramaeconomico.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     OdontogramaEconomico:
 *       type: object
 *       required:
 *         - FechaCreacion
 *         - FechayHoraFin
 *         - Especificaciones
 *         - Observaciones
 *         - IngresoId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Odontograma Economico
 *         FechaCreacion:
 *           type: Date
 *           description: Fecha de Creacion del Odontograma Economico
 *         Especificaciones:
 *           type: string
 *           description: Especificaciones del Odontograma Economico
 *         Observaciones:
 *           type: string
 *           description: Observaciones del Odontograma Economico
 *         IngresoId:
 *           type: number
 *           description: Identificador del Ingreso
 *         PacienteId:
 *           type: number
 *           description: Identificador de la Historia Clinica
 *       example:
 *         id: "1"
 *         FechaCreacion: "Odontopediatría"
 *         Especificaciones: "Odontopediatría"
 *         Observaciones: "Odontopediatría"
 *         IngresoId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: OdontogramaEconomico
 *   description: API para gestionar OdontogramaEconomico
 */
/**
 * @swagger
 * /odontograma/odontogramaeconomico:
 *   get:
 *     summary: Obtiene la lista de OdontogramaEconomico
 *     tags: [OdontogramaEconomico]
 *     responses:
 *       200:
 *         description: Lista de OdontogramaEconomico.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/OdontogramaEconomico'
 */
router.get("/", verifyToken, OdontogramaEconomicoController.getOdontogramaEconomicoAll);
/**
 * @swagger
 * /odontograma/odontogramaeconomico/{id}:
 *   get:
 *     summary: Obtiene un OdontogramaEconomico por ID
 *     tags: [OdontogramaEconomico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OdontogramaEconomico
 *     responses:
 *       200:
 *         description: OdontogramaEconomico encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OdontogramaEconomico'
 *       404:
 *         description: OdontogramaEconomico no encontrado.
 */
router.get("/:id", verifyToken, OdontogramaEconomicoController.getOdontogramaEconomicoById);
/**
 * @swagger
 * /odontograma/odontogramaeconomico/create:
 *   post:
 *     summary: Crea un nuevo OdontogramaEconomico
 *     tags: [OdontogramaEconomico]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OdontogramaEconomico'
 *     responses:
 *       201:
 *         description: OdontogramaEconomico creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OdontogramaEconomico'
 */
router.post("/create", verifyToken, OdontogramaEconomicoController.createOdontogramaEconomico); // Aplicamos validación
/**
 * @swagger
 * /odontograma/odontogramaeconomico/update/{id}:
 *   put:
 *     summary: Actualiza un OdontogramaEconomico existente
 *     tags: [OdontogramaEconomico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OdontogramaEconomico a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OdontogramaEconomico'
 *     responses:
 *       200:
 *         description: OdontogramaEconomico actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OdontogramaEconomico'
 *       404:
 *         description: OdontogramaEconomico no encontrado.
 */
router.put("/update/:id", verifyToken, OdontogramaEconomicoController.updateOdontogramaEconomico);
/**
 * @swagger
 * /odontograma/odontogramaeconomico/{id}:
 *   delete:
 *     summary: Elimina un OdontogramaEconomico por ID
 *     tags: [OdontogramaEconomico]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OdontogramaEconomico a eliminar
 *     responses:
 *       200:
 *         description: OdontogramaEconomico eliminado exitosamente.
 *       404:
 *         description: OdontogramaEconomico no encontrado.
 */
router.delete("/:id", verifyToken, OdontogramaEconomicoController.deleteOdontogramaEconomico);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /odontograma/odontogramaeconomico/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un OdontogramaEconomico por paciente
 *     tags: [OdontogramaEconomico]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: OdontogramaEconomico encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OdontogramaEconomico'
 *       404:
 *         description: OdontogramaEconomico no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, OdontogramaEconomicoController.getOdontogramaEconomicoByPaciente);
/**
 * @swagger
 * /odontograma/odontogramaeconomico/search/factura/{factura}:
 *   get:
 *     summary: Obtiene un OdontogramaEconomico por factura
 *     tags: [OdontogramaEconomico]
 *     parameters:
 *       - in: path
 *         name: factura
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador de la factura
 *     responses:
 *       200:
 *         description: OdontogramaEconomico encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OdontogramaEconomico'
 *       404:
 *         description: OdontogramaEconomico no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/search/factura/:factura", verifyToken, OdontogramaEconomicoController.getOdontogramaEconomicoByFactura);

module.exports = router;