const express = require("express");
const OrdenLaboratorioController = require("../controllers/ordenlaboratorio.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     OrdenLaboratorio:
 *       type: object
 *       required:
 *         - FechaOrden
 *         - Observaciones
 *         - EstadoOrdenId
 *         - PacienteId
 *         - UsuarioId
 *         - PersonalId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del OrdenLaboratorio
 *         FechaOrden:
 *           type: string
 *           description: Fecha Orden del OrdenLaboratorio
 *         Observaciones:
 *           type: string
 *           description: Observaciones del OrdenLaboratorio
 *         EstadoOrdenId:
 *           type: number
 *           description: Estado Orden del OrdenLaboratorio (solicitado, respuesta, etc)
 *         PacienteId:
 *           type: number
 *           description: Historia Clinica del OrdenLaboratorio
 *         UsuarioId:
 *           type: number
 *           description: Usuario del OrdenLaboratorio (personal que registra)
 *         PersonalId:
 *           type: number
 *           description: Personal del OrdenLaboratorio (personal que atiende)
 *       example:
 *         id: "1"
 *         FechaOrden: "2025-12-11"
 *         Observaciones: "Ninguna"
 *         EstadoOrdenId: "1"
 *         PacienteId: "1"
 *         UsuarioId: "1"
 *         PersonalId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: OrdenLaboratorio
 *   description: API para gestionar OrdenLaboratorio
 */
/**
 * @swagger
 * /laboratorio/orden:
 *   get:
 *     summary: Obtiene la lista de OrdenLaboratorio
 *     tags: [OrdenLaboratorio]
 *     responses:
 *       200:
 *         description: Lista de OrdenLaboratorio.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/OrdenLaboratorio'
 */
router.get("/", verifyToken, OrdenLaboratorioController.getOrdenLaboratorioAll);
/**
 * @swagger
 * /laboratorio/orden/{id}:
 *   get:
 *     summary: Obtiene un OrdenLaboratorio por ID
 *     tags: [OrdenLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OrdenLaboratorio
 *     responses:
 *       200:
 *         description: OrdenLaboratorio encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OrdenLaboratorio'
 *       404:
 *         description: OrdenLaboratorio no encontrado.
 */
router.get("/:id", verifyToken, OrdenLaboratorioController.getOrdenLaboratorioById);
/**
 * @swagger
 * /laboratorio/orden/create:
 *   post:
 *     summary: Crea un nuevo OrdenLaboratorio
 *     tags: [OrdenLaboratorio]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OrdenLaboratorio'
 *     responses:
 *       201:
 *         description: OrdenLaboratorio creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OrdenLaboratorio'
 */
router.post("/create", verifyToken, OrdenLaboratorioController.createOrdenLaboratorio);
/**
 * @swagger
 * /laboratorio/orden/update/{id}:
 *   put:
 *     summary: Actualiza un OrdenLaboratorio existente
 *     tags: [OrdenLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OrdenLaboratorio a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OrdenLaboratorio'
 *     responses:
 *       200:
 *         description: OrdenLaboratorio actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OrdenLaboratorio'
 *       404:
 *         description: OrdenLaboratorio no encontrado.
 */
router.put("/update/:id", verifyToken, OrdenLaboratorioController.updateOrdenLaboratorio);
/**
 * @swagger
 * /laboratorio/orden/{id}:
 *   delete:
 *     summary: Elimina un OrdenLaboratorio por ID
 *     tags: [OrdenLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OrdenLaboratorio a eliminar
 *     responses:
 *       200:
 *         description: OrdenLaboratorio eliminado exitosamente.
 *       404:
 *         description: OrdenLaboratorio no encontrado.
 */
router.delete("/:id", verifyToken, OrdenLaboratorioController.deleteOrdenLaboratorio);
/**
 * @swagger
 * /laboratorio/orden/filtro/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un OrdenLaboratorio por paciente
 *     tags: [OrdenLaboratorio]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: OrdenLaboratorioes del paciente encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OrdenLaboratorio'
 *       404:
 *         description: OrdenLaboratorio no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/paciente/:paciente", verifyToken, OrdenLaboratorioController.getOrdenLaboratorioByPaciente);

module.exports = router;
