const express = require("express");
const OrdenRadiografiaController = require("../controllers/ordenradiografia.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     OrdenRadiografia:
 *       type: object
 *       required:
 *         - FechaOrden
 *         - Observaciones
 *         - EstadoOrdenId
 *         - PacienteId
 *         - UsuarioId
 *         - PersonalId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del OrdenRadiografia
 *         FechaOrden:
 *           type: string
 *           description: Fecha Orden del OrdenRadiografia
 *         Observaciones:
 *           type: string
 *           description: Observaciones del OrdenRadiografia
 *         EstadoOrdenId:
 *           type: string
 *           description: Estado Orden del OrdenRadiografia
 *         PacienteId:
 *           type: string
 *           description: Historia Clinica del OrdenRadiografia
 *         UsuarioId:
 *           type: string
 *           description: Usuario del OrdenRadiografia
 *         PersonalId:
 *           type: string
 *           description: Personal del OrdenRadiografia
 *       example:
 *         id: "1"
 *         FechaOrden: "2025-04-12"
 *         Observaciones: "Ninguna"
 *         EstadoOrdenId: "1"
 *         PacienteId: "1"
 *         UsuarioId: "1"
 *         PersonalId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: OrdenRadiografia
 *   description: API para gestionar OrdenRadiografia
 */
/**
 * @swagger
 * /radiografia/orden:
 *   get:
 *     summary: Obtiene la lista de OrdenRadiografia
 *     tags: [OrdenRadiografia]
 *     responses:
 *       200:
 *         description: Lista de OrdenRadiografia.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/OrdenRadiografia'
 */
router.get("/", verifyToken, OrdenRadiografiaController.getOrdenRadiografiaAll);
/**
 * @swagger
 * /radiografia/orden/{id}:
 *   get:
 *     summary: Obtiene un OrdenRadiografia por ID
 *     tags: [OrdenRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OrdenRadiografia
 *     responses:
 *       200:
 *         description: OrdenRadiografia encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OrdenRadiografia'
 *       404:
 *         description: OrdenRadiografia no encontrado.
 */
router.get("/:id", verifyToken, OrdenRadiografiaController.getOrdenRadiografiaById);
/**
 * @swagger
 * /radiografia/orden/create:
 *   post:
 *     summary: Crea un nuevo OrdenRadiografia
 *     tags: [OrdenRadiografia]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OrdenRadiografia'
 *     responses:
 *       201:
 *         description: OrdenRadiografia creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OrdenRadiografia'
 */
router.post("/create", verifyToken, OrdenRadiografiaController.createOrdenRadiografia);
/**
 * @swagger
 * /radiografia/orden/update/{id}:
 *   put:
 *     summary: Actualiza un OrdenRadiografia existente
 *     tags: [OrdenRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OrdenRadiografia a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OrdenRadiografia'
 *     responses:
 *       200:
 *         description: OrdenRadiografia actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OrdenRadiografia'
 *       404:
 *         description: OrdenRadiografia no encontrado.
 */
router.put("/update/:id", verifyToken, OrdenRadiografiaController.updateOrdenRadiografia);
/**
 * @swagger
 * /radiografia/orden/{id}:
 *   delete:
 *     summary: Elimina un OrdenRadiografia por ID
 *     tags: [OrdenRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del OrdenRadiografia a eliminar
 *     responses:
 *       200:
 *         description: OrdenRadiografia eliminado exitosamente.
 *       404:
 *         description: OrdenRadiografia no encontrado.
 */
router.delete("/:id", verifyToken, OrdenRadiografiaController.deleteOrdenRadiografia);
/**
 * @swagger
 * /radiografia/orden/filtro/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un OrdenRadiografia por paciente
 *     tags: [OrdenRadiografia]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: OrdenRadiografiaes del paciente encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OrdenRadiografia'
 *       404:
 *         description: OrdenRadiografia no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/paciente/:paciente", verifyToken, OrdenRadiografiaController.getOrdenRadiografiaByPaciente);

module.exports = router;
