const express = require("express");
const pagoController = require("../controllers/pago.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Pago:
 *       type: object
 *       required:
 *         - Fecha
 *         - PersonalId
 *         - PeriodoId
 *         - EgresoId
 *         - Monto
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Pago
 *         Fecha:
 *           type: Date
 *           description: Fecha del Pago
 *         Monto:
 *           type: string
 *           description: Monto pagado al Trabajador
 *         PersonalId:
 *           type: number
 *           description: Personal que recibe el Pago
 *         PeriodoId:
 *           type: number
 *           description: Periodo de la Paga
 *         EgresoId:
 *           type: number
 *           description: Egreso salido de la lista de Egresos
 *       example:
 *         id: "1"
 *         Fecha: "2025-12-31"
 *         Monto: "24325"
 *         PersonalId: "1"
 *         PeriodoId: "1"
 *         EgresoId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Pago
 *   description: API para gestionar Pagos
 */
/**
 * @swagger
 * /rrhh/pago:
 *   get:
 *     summary: Obtiene la lista de Pagos
 *     tags: [Pago]
 *     responses:
 *       200:
 *         description: Lista de Pagos.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Pago'
 */
router.get("/", verifyToken, pagoController.getPagoAll);
/**
 * @swagger
 * /rrhh/pago/trabajador/{trabajador}:
 *   get:
 *     summary: Obtiene Pagos por trabajador
 *     tags: [Pago]
 *     parameters:
 *       - in: path
 *         name: trabajador
 *         schema:
 *           type: number
 *         required: true
 *         description: Pagos del trabajador
 *     responses:
 *       200:
 *         description: Pagos encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Pago'
 *       404:
 *         description: Pago no encontrado.
 */
router.get("/trabajador/:trabajador", verifyToken, pagoController.getPagoByTrabajador);
/**
 * @swagger
 * /rrhh/pago/egreso/{egreso}:
 *   get:
 *     summary: Obtiene Pagos por trabajador
 *     tags: [Pago]
 *     parameters:
 *       - in: path
 *         name: egreso
 *         schema:
 *           type: number
 *         required: true
 *         description: Codigo de egreso del trabajador
 *     responses:
 *       200:
 *         description: Pagos encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Pago'
 *       404:
 *         description: Pago no encontrado.
 */
router.get("/egreso/:egreso", verifyToken, pagoController.getPagoByEgreso);
/**
 * @swagger
 * /rrhh/pago/{id}:
 *   get:
 *     summary: Obtiene un Pago por ID
 *     tags: [Pago]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Pago
 *     responses:
 *       200:
 *         description: Pago encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Pago'
 *       404:
 *         description: Pago no encontrado.
 */
router.get("/:id", verifyToken, pagoController.getPagoById);
/**
 * @swagger
 * /rrhh/pago/create:
 *   post:
 *     summary: Crea un nuevo Pago
 *     tags: [Pago]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Pago'
 *     responses:
 *       201:
 *         description: Pago creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Pago'
 */
router.post("/create", verifyToken, pagoController.createPago);
/**
 * @swagger
 * /rrhh/pago/update/{id}:
 *   put:
 *     summary: Actualiza un Pago existente
 *     tags: [Pago]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Pago a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Pago'
 *     responses:
 *       200:
 *         description: Pago actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Pago'
 *       404:
 *         description: Pago no encontrado.
 */
router.put("/update/:id", verifyToken, pagoController.updatePago);
/**
 * @swagger
 * /rrhh/pago/{id}:
 *   delete:
 *     summary: Elimina un Pago por ID
 *     tags: [Pago]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Pago a eliminar
 *     responses:
 *       200:
 *         description: Pago eliminado exitosamente.
 *       404:
 *         description: Pago no encontrado.
 */
router.delete("/:id", verifyToken, pagoController.deletePago);

// Rutas de filtrado individual


module.exports = router;
