const express = require("express");
const { verifyToken } = require('../middlewares/auth.middleware');
const patrimonioController = require("../controllers/patrimonio.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Patrimonio:
 *       type: object
 *       required:
 *         - Codigo
 *         - Nombre
 *         - Descripcion
 *         - Ubicacion
 *         - FechaAdquisicion
 *         - FechaEliminacion
 *         - Estado
 *         - Valor
 *         - Responsable
 *         - SedeId
 *         - SubCategoriaPatrimonioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Patrimonio
 *         Codigo:
 *           type: string
 *           description: Codigo del Bien Patrimonial
 *         Nombre:
 *           type: string
 *           description: Nombre del Bien Patrimonial
 *         Descripcion:
 *           type: string
 *           description: Descripcion del Bien Patrimonial
 *         Ubicacion:
 *           type: string
 *           description: Ubicacion del Bien Patrimonial
 *         FechaAdquisicion:
 *           type: Date
 *           description: Fecha de Adquisicion del Bien Patrimonial
 *         FechaEliminacion:
 *           type: string
 *           description: Fecha de Eliminacion del Bien Patrimonial
 *         Estado:
 *           type: string
 *           description: Estado del Bien Patrimonial
 *         Valor:
 *           type: string
 *           description: Valor del Bien Patrimonial
 *         Responsable:
 *           type: string
 *           description: Responsable del Bien Patrimonial
 *         SedeId:
 *           type: number
 *           description: Sede a la que pertenece el Bien Patrimonial
 *         SubCategoriaPatrimonioId:
 *           type: number
 *           description: Sub Categoria a la que pertenece el Bien Patrimonial
 *       example:
 *         id: "1"
 *         Codigo: "ECM-21321"
 *         Nombre: "Monitor LG 24 Pulgadas"
 *         Descripcion: "Monitor LG 24 Pulgadas fdasfdsaf"
 *         Ubicacion: "Oficina de Comunicaciones"
 *         FechaAdquisicion: "2020-12-12"
 *         FechaEliminacion: "2026-12-12"
 *         Estado: "En uso"
 *         Valor: "8/10"
 *         Responsable: "Jefe de Comunicaciones"
 *         SedeId: "1"
 *         SubCategoriaPatrimonioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Patrimonio
 *   description: API para gestionar Patrimonios
 */
/**
 * @swagger
 * /patrimonio/patrimonio:
 *   get:
 *     summary: Obtiene la lista de Patrimonios
 *     tags: [Patrimonio]
 *     responses:
 *       200:
 *         description: Lista de Patrimonios.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Patrimonio'
 */
router.get("/", verifyToken, patrimonioController.getPatrimonioAll);
/**
 * @swagger
 * /patrimonio/patrimonio/subcategoria/{subcategoria}:
 *   get:
 *     summary: Obtiene un Patrimonio por subcategoria
 *     tags: [Patrimonio]
 *     parameters:
 *       - in: path
 *         name: subcategoria
 *         schema:
 *           type: number
 *         required: true
 *         description: subcategoria del Patrimonio
 *     responses:
 *       200:
 *         description: Patrimonios encontrados por subcategoria
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Patrimonio'
 *       404:
 *         description: Patrimonio no encontrado.
 */
router.get("/subcategoria/:sede/:subcategoria", verifyToken, patrimonioController.getPatrimonioABySubCategoria);
/**
 * @swagger
 * /patrimonio/patrimonio/sede/{sede}:
 *   get:
 *     summary: Obtiene un Patrimonio por sede
 *     tags: [Patrimonio]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede del Patrimonio
 *     responses:
 *       200:
 *         description: Patrimonios encontrados por sede
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Patrimonio'
 *       404:
 *         description: Patrimonio no encontrado.
 */
router.get("/sede/:sede", verifyToken, patrimonioController.getPatrimonioABySede);
/**
 * @swagger
 * /patrimonio/patrimonio/estado/{estado}:
 *   get:
 *     summary: Obtiene un Patrimonio por estado
 *     tags: [Patrimonio]
 *     parameters:
 *       - in: path
 *         name: estado
 *         schema:
 *           type: number
 *         required: true
 *         description: estado del Patrimonio
 *     responses:
 *       200:
 *         description: Patrimonios encontrados por estado
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Patrimonio'
 *       404:
 *         description: Patrimonio no encontrado.
 */
router.get("/estado/:sede/:estado", verifyToken, patrimonioController.getPatrimonioAByEstado);
/**
 * @swagger
 * /patrimonio/patrimonio/resposable/{resposable}:
 *   get:
 *     summary: Obtiene un Patrimonio por resposable
 *     tags: [Patrimonio]
 *     parameters:
 *       - in: path
 *         name: resposable
 *         schema:
 *           type: number
 *         required: true
 *         description: resposable del Patrimonio
 *     responses:
 *       200:
 *         description: Patrimonios encontrados por Resposable
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Patrimonio'
 *       404:
 *         description: Patrimonio no encontrado.
 */
router.get("/resposable/:sede/:resposable", verifyToken, patrimonioController.getPatrimonioAByResponsable);
/**
 * @swagger
 * /patrimonio/patrimonio/codigo/{codigo}:
 *   get:
 *     summary: Obtiene un Patrimonio por codigo
 *     tags: [Patrimonio]
 *     parameters:
 *       - in: path
 *         name: codigo
 *         schema:
 *           type: number
 *         required: true
 *         description: codigo del Patrimonio
 *     responses:
 *       200:
 *         description: Patrimonio encontrado por codigo
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Patrimonio'
 *       404:
 *         description: Patrimonio no encontrado.
 */
router.get("/codigo/:codigo", verifyToken, patrimonioController.getPatrimonioAByCodigo);
/**
 * @swagger
 * /patrimonio/patrimonio/{id}:
 *   get:
 *     summary: Obtiene un Patrimonio por ID
 *     tags: [Patrimonio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Patrimonio
 *     responses:
 *       200:
 *         description: Patrimonio encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Patrimonio'
 *       404:
 *         description: Patrimonio no encontrado.
 */
router.get("/:id", verifyToken, patrimonioController.getPatrimonioById);
/**
 * @swagger
 * /patrimonio/patrimonio/create:
 *   post:
 *     summary: Crea un nuevo Patrimonio
 *     tags: [Patrimonio]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Patrimonio'
 *     responses:
 *       201:
 *         description: Patrimonio creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Patrimonio'
 */
router.post("/create", verifyToken, patrimonioController.createPatrimonio);
/**
 * @swagger
 * /patrimonio/patrimonio/update/{id}:
 *   put:
 *     summary: Actualiza un Patrimonio existente
 *     tags: [Patrimonio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Patrimonio a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Patrimonio'
 *     responses:
 *       200:
 *         description: Patrimonio actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Patrimonio'
 *       404:
 *         description: Patrimonio no encontrado.
 */
router.put("/update/:id", verifyToken, patrimonioController.updatePatrimonio);
/**
 * @swagger
 * /patrimonio/patrimonio/{id}:
 *   delete:
 *     summary: Elimina un Patrimonio por ID
 *     tags: [Patrimonio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Patrimonio a eliminar
 *     responses:
 *       200:
 *         description: Patrimonio eliminado exitosamente.
 *       404:
 *         description: Patrimonio no encontrado.
 */
router.delete("/:id", verifyToken, patrimonioController.deletePatrimonio);

// Rutas de filtrado individual


module.exports = router;
