const express = require("express");
const PeticionLaboratorioController = require("../controllers/peticionlaboratorio.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     PeticionLaboratorio:
 *       type: object
 *       required:
 *         - Nombre
 *         - OrdenLaboratorioId
 *         - ItemLaboratorioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del PeticionLaboratorio
 *         Nombre:
 *           type: string
 *           description: Nombre del PeticionLaboratorio
 *         OrdenLaboratorioId:
 *           type: number
 *           description: Relacion del PeticionLaboratorio con la Orden Laboratorio
 *         ItemLaboratorioId:
 *           type: number
 *           description: Relacion del PeticionLaboratorio con el Item Laboratorio
 *       example:
 *         id: "1"
 *         Nombre: "sdafasf"
 *         OrdenLaboratorioId: "1"
 *         ItemLaboratorioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: PeticionLaboratorio
 *   description: API para gestionar PeticionLaboratorio
 */
/**
 * @swagger
 * /laboratorio/peticion:
 *   get:
 *     summary: Obtiene la lista de PeticionLaboratorio
 *     tags: [PeticionLaboratorio]
 *     responses:
 *       200:
 *         description: Lista de PeticionLaboratorio.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/PeticionLaboratorio'
 */
router.get("/", verifyToken, PeticionLaboratorioController.getPeticionLaboratorioAll);
/**
 * @swagger
 * /laboratorio/peticion/{id}:
 *   get:
 *     summary: Obtiene un PeticionLaboratorio por ID
 *     tags: [PeticionLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PeticionLaboratorio
 *     responses:
 *       200:
 *         description: PeticionLaboratorio encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionLaboratorio'
 *       404:
 *         description: PeticionLaboratorio no encontrado.
 */
router.get("/:id", verifyToken, PeticionLaboratorioController.getPeticionLaboratorioById);
/**
 * @swagger
 * /laboratorio/peticion/create:
 *   post:
 *     summary: Crea un nuevo PeticionLaboratorio
 *     tags: [PeticionLaboratorio]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PeticionLaboratorio'
 *     responses:
 *       201:
 *         description: PeticionLaboratorio creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionLaboratorio'
 */
router.post("/create", verifyToken, PeticionLaboratorioController.createPeticionLaboratorio);
/**
 * @swagger
 * /laboratorio/peticion/update/{id}:
 *   put:
 *     summary: Actualiza un PeticionLaboratorio existente
 *     tags: [PeticionLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PeticionLaboratorio a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PeticionLaboratorio'
 *     responses:
 *       200:
 *         description: PeticionLaboratorio actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionLaboratorio'
 *       404:
 *         description: PeticionLaboratorio no encontrado.
 */
router.put("/update/:id", verifyToken, PeticionLaboratorioController.updatePeticionLaboratorio);
/**
 * @swagger
 * /laboratorio/peticion/{id}:
 *   delete:
 *     summary: Elimina un PeticionLaboratorio por ID
 *     tags: [PeticionLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PeticionLaboratorio a eliminar
 *     responses:
 *       200:
 *         description: PeticionLaboratorio eliminado exitosamente.
 *       404:
 *         description: PeticionLaboratorio no encontrado.
 */
router.delete("/:id", verifyToken, PeticionLaboratorioController.deletePeticionLaboratorio);
/**
 * @swagger
 * /laboratorio/peticion/filtro/item/{item}:
 *   get:
 *     summary: Obtiene un PeticionLaboratorio por item
 *     tags: [PeticionLaboratorio]
 *     parameters:
 *       - in: path
 *         name: item
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del item
 *     responses:
 *       200:
 *         description: PeticionLaboratorioes del item encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionLaboratorio'
 *       404:
 *         description: PeticionLaboratorio no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/item/:item", verifyToken, PeticionLaboratorioController.getPeticionLaboratorioByItemLaboratorio);
/**
 * @swagger
 * /laboratorio/peticion/filtro/orden/{orden}:
 *   get:
 *     summary: Obtiene un PeticionLaboratorio por orden
 *     tags: [PeticionLaboratorio]
 *     parameters:
 *       - in: path
 *         name: orden
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del orden
 *     responses:
 *       200:
 *         description: PeticionLaboratorioes del orden encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionLaboratorio'
 *       404:
 *         description: PeticionLaboratorio no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/orden/:orden", verifyToken, PeticionLaboratorioController.getPeticionLaboratorioByOrdenLaboratorio);

/**
 * @swagger
 * /laboratorio/peticion/filtro/orden/{orden}:
 *   get:
 *     summary: Obtiene un PeticionLaboratorio por orden
 *     tags: [PeticionLaboratorio]
 *     parameters:
 *       - in: path
 *         name: orden
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del orden
 *     responses:
 *       200:
 *         description: PeticionLaboratorioes del orden encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionLaboratorio'
 *       404:
 *         description: PeticionLaboratorio no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/search/itemyorden/:orden/:item", verifyToken, PeticionLaboratorioController.getPeticionLaboratorioByOrdeneItem);

module.exports = router;
