const express = require("express");
const PeticionRadiografiaController = require("../controllers/peticionradiografia.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     PeticionRadiografia:
 *       type: object
 *       required:
 *         - Nombre
 *         - Extras
 *         - ObservacionEspecifica
 *         - OrdenRadiografiaId
 *         - ItemRadiografiaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del PeticionRadiografia
 *         Nombre:
 *           type: string
 *           description: Nombre del PeticionRadiografia
 *         Extras:
 *           type: string
 *           description: Extras del PeticionRadiografia
 *         ObservacionEspecifica:
 *           type: string
 *           description: Observacion Especifica del PeticionRadiografia
 *         OrdenRadiografiaId:
 *           type: string
 *           description: Orden Radiografia del PeticionRadiografia
 *         ItemRadiografiaId:
 *           type: string
 *           description: Item Radiografia del PeticionRadiografia
 *       example:
 *         id: "1"
 *         Nombre: "Frontal y Lateral"
 *         Extras: "Pieza 2, Pieza 7, etc"
 *         ObservacionEspecifica: "Ninguna"
 *         OrdenRadiografiaId: "1"
 *         ItemRadiografiaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: PeticionRadiografia
 *   description: API para gestionar PeticionRadiografia
 */
/**
 * @swagger
 * /radiografia/peticion:
 *   get:
 *     summary: Obtiene la lista de PeticionRadiografia
 *     tags: [PeticionRadiografia]
 *     responses:
 *       200:
 *         description: Lista de PeticionRadiografia.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/PeticionRadiografia'
 */
router.get("/", verifyToken, PeticionRadiografiaController.getPeticionRadiografiaAll);
/**
 * @swagger
 * /radiografia/peticion/{id}:
 *   get:
 *     summary: Obtiene un PeticionRadiografia por ID
 *     tags: [PeticionRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PeticionRadiografia
 *     responses:
 *       200:
 *         description: PeticionRadiografia encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionRadiografia'
 *       404:
 *         description: PeticionRadiografia no encontrado.
 */
router.get("/:id", verifyToken, PeticionRadiografiaController.getPeticionRadiografiaById);
/**
 * @swagger
 * /radiografia/peticion/create:
 *   post:
 *     summary: Crea un nuevo PeticionRadiografia
 *     tags: [PeticionRadiografia]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PeticionRadiografia'
 *     responses:
 *       201:
 *         description: PeticionRadiografia creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionRadiografia'
 */
router.post("/create", verifyToken, PeticionRadiografiaController.createPeticionRadiografia);
/**
 * @swagger
 * /radiografia/peticion/update/{id}:
 *   put:
 *     summary: Actualiza un PeticionRadiografia existente
 *     tags: [PeticionRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PeticionRadiografia a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PeticionRadiografia'
 *     responses:
 *       200:
 *         description: PeticionRadiografia actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionRadiografia'
 *       404:
 *         description: PeticionRadiografia no encontrado.
 */
router.put("/update/:id", verifyToken, PeticionRadiografiaController.updatePeticionRadiografia);
/**
 * @swagger
 * /radiografia/peticion/{id}:
 *   delete:
 *     summary: Elimina un PeticionRadiografia por ID
 *     tags: [PeticionRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PeticionRadiografia a eliminar
 *     responses:
 *       200:
 *         description: PeticionRadiografia eliminado exitosamente.
 *       404:
 *         description: PeticionRadiografia no encontrado.
 */
router.delete("/:id", verifyToken, PeticionRadiografiaController.deletePeticionRadiografia);
/**
 * @swagger
 * /radiografia/peticion/filtro/item/{item}:
 *   get:
 *     summary: Obtiene un PeticionRadiografia por item
 *     tags: [PeticionRadiografia]
 *     parameters:
 *       - in: path
 *         name: item
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del item
 *     responses:
 *       200:
 *         description: PeticionRadiografiaes del item encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionRadiografia'
 *       404:
 *         description: PeticionRadiografia no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/item/:item", verifyToken, PeticionRadiografiaController.getPeticionRadiografiaByItemRadiografia);
/**
 * @swagger
 * /radiografia/peticion/filtro/orden/{orden}:
 *   get:
 *     summary: Obtiene un PeticionRadiografia por orden
 *     tags: [PeticionRadiografia]
 *     parameters:
 *       - in: path
 *         name: orden
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del orden
 *     responses:
 *       200:
 *         description: PeticionRadiografiaes del orden encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PeticionRadiografia'
 *       404:
 *         description: PeticionRadiografia no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/orden/:orden", verifyToken, PeticionRadiografiaController.getPeticionRadiografiaByOrdenRadiografia);

module.exports = router;
