const express = require("express");
const PlanTratamientoController = require("../controllers/plantratamiento.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     PlanTratamiento:
 *       type: object
 *       required:
 *         - FechaInicio
 *         - Total
 *         - AgendamientoId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del PlanTratamiento
 *         FechaInicio:
 *           type: Date
 *           description: Fecha Inicio del Plan Tratamiento
 *         Total:
 *           type: string
 *           description: Monto Sumando del plan de tratamiento
 *         AgendamientoId:
 *           type: number
 *           description: Identificador del Agendamiento
 *         PacienteId:
 *           type: number
 *           description: Identificador de la Historia Clinica
 *       example:
 *         id: "1"
 *         FechaInicio: "2025-12-11"
 *         Total: "214214"
 *         AgendamientoId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: PlanTratamiento
 *   description: API para gestionar PlanTratamiento
 */
/**
 * @swagger
 * /paciente/plantratamiento:
 *   get:
 *     summary: Obtiene la lista de PlanTratamiento
 *     tags: [PlanTratamiento]
 *     responses:
 *       200:
 *         description: Lista de PlanTratamiento.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/PlanTratamiento'
 */
router.get("/", verifyToken, PlanTratamientoController.getPlanTratamientoAll);
/**
 * @swagger
 * /paciente/plantratamiento/{id}:
 *   get:
 *     summary: Obtiene un PlanTratamiento por ID
 *     tags: [PlanTratamiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PlanTratamiento
 *     responses:
 *       200:
 *         description: PlanTratamiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PlanTratamiento'
 *       404:
 *         description: PlanTratamiento no encontrado.
 */
router.get("/:id", verifyToken, PlanTratamientoController.getPlanTratamientoById);
/**
 * @swagger
 * /paciente/plantratamiento/create:
 *   post:
 *     summary: Crea un nuevo PlanTratamiento
 *     tags: [PlanTratamiento]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PlanTratamiento'
 *     responses:
 *       201:
 *         description: PlanTratamiento creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PlanTratamiento'
 */
router.post("/create", verifyToken, PlanTratamientoController.createPlanTratamiento); // Aplicamos validación
/**
 * @swagger
 * /paciente/plantratamiento/update/{id}:
 *   put:
 *     summary: Actualiza un PlanTratamiento existente
 *     tags: [PlanTratamiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PlanTratamiento a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PlanTratamiento'
 *     responses:
 *       200:
 *         description: PlanTratamiento actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PlanTratamiento'
 *       404:
 *         description: PlanTratamiento no encontrado.
 */
router.put("/update/:id", verifyToken, PlanTratamientoController.updatePlanTratamiento);
/**
 * @swagger
 * /paciente/plantratamiento/{id}:
 *   delete:
 *     summary: Elimina un PlanTratamiento por ID
 *     tags: [PlanTratamiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PlanTratamiento a eliminar
 *     responses:
 *       200:
 *         description: PlanTratamiento eliminado exitosamente.
 *       404:
 *         description: PlanTratamiento no encontrado.
 */
router.delete("/:id", verifyToken, PlanTratamientoController.deletePlanTratamiento);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/plantratamiento/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un PlanTratamiento por paciente
 *     tags: [PlanTratamiento]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: PlanTratamiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PlanTratamiento'
 *       404:
 *         description: PlanTratamiento no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, PlanTratamientoController.getPlanTratamientoByPaciente);

module.exports = router;