const express = require("express");
const PresupuestoParcialController = require("../controllers/presupuestoparcial.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     PresupuestoParcial:
 *       type: object
 *       required:
 *         - ACuenta
 *         - Saldo
 *         - EF
 *         - TC
 *         - Fecha
 *         - PacienteId
 *         - IngresoId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Presupuesto Parcial
 *         ACuenta:
 *           type: number
 *           description: ACuenta del Presupuesto Parcial
 *         Saldo:
 *           type: number
 *           description: Saldo del Presupuesto Parcial
 *         EF:
 *           type: string
 *           description: EF del Presupuesto Parcial
 *         TC:
 *           type: string
 *           description: TC del Presupuesto Parcial
 *         Fecha:
 *           type: Date
 *           description: Fecha del Presupuesto Parcial
 *         PacienteId:
 *           type: number
 *           description: Indentificador de la Historia Clinica
 *         IngresoId:
 *           type: number
 *           description: Indentificador del Ingreso
 *       example:
 *         id: "1"
 *         ACuenta: "124214"
 *         Saldo: "10214"
 *         EF: "fdasfsdf"
 *         TC: "fdsfasdf"
 *         Fecha: "2025-11-21"
 *         PacienteId: "1"
 *         IngresoId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: PresupuestoParcial
 *   description: API para gestionar PresupuestoParcial
 */
/**
 * @swagger
 * /economia/presupuestoparcial:
 *   get:
 *     summary: Obtiene la lista de PresupuestoParcial
 *     tags: [PresupuestoParcial]
 *     responses:
 *       200:
 *         description: Lista de PresupuestoParcial.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/PresupuestoParcial'
 */
router.get("/", verifyToken, PresupuestoParcialController.getPresupuestoParcialAll);
/**
 * @swagger
 * /economia/presupuestoparcial/{id}:
 *   get:
 *     summary: Obtiene un PresupuestoParcial por ID
 *     tags: [PresupuestoParcial]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PresupuestoParcial
 *     responses:
 *       200:
 *         description: PresupuestoParcial encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PresupuestoParcial'
 *       404:
 *         description: PresupuestoParcial no encontrado.
 */
router.get("/:id", verifyToken, PresupuestoParcialController.getPresupuestoParcialById);
/**
 * @swagger
 * /economia/presupuestoparcial/create:
 *   post:
 *     summary: Crea un nuevo PresupuestoParcial
 *     tags: [PresupuestoParcial]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PresupuestoParcial'
 *     responses:
 *       201:
 *         description: PresupuestoParcial creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PresupuestoParcial'
 */
router.post("/create", verifyToken, PresupuestoParcialController.createPresupuestoParcial);
/**
 * @swagger
 * /economia/presupuestoparcial/update/{id}:
 *   put:
 *     summary: Actualiza un PresupuestoParcial existente
 *     tags: [PresupuestoParcial]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PresupuestoParcial a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PresupuestoParcial'
 *     responses:
 *       200:
 *         description: PresupuestoParcial actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PresupuestoParcial'
 *       404:
 *         description: PresupuestoParcial no encontrado.
 */
router.put("/update/:id", verifyToken, PresupuestoParcialController.updatePresupuestoParcial);
/**
 * @swagger
 * /economia/presupuestoparcial/{id}:
 *   delete:
 *     summary: Elimina un PresupuestoParcial por ID
 *     tags: [PresupuestoParcial]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PresupuestoParcial a eliminar
 *     responses:
 *       200:
 *         description: PresupuestoParcial eliminado exitosamente.
 *       404:
 *         description: PresupuestoParcial no encontrado.
 */
router.delete("/:id", verifyToken, PresupuestoParcialController.deletePresupuestoParcial);
/**
 * @swagger
 * /economia/presupuestoparcial/historia/{historia}:
 *   get:
 *     summary: Obtiene un PresupuestoParcial por usuario
 *     tags: [PresupuestoParcial]
 *     parameters:
 *       - in: path
 *         name: historia
 *         schema:
 *           type: number
 *         required: true
 *         description: historia del Presupuesto Parcial
 *     responses:
 *       200:
 *         description: Presupuesto Parcial encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PresupuestoParcial'
 *       404:
 *         description: PresupuestoParcial no encontrado.
 */
router.get("/filtrar/historia/:historia", verifyToken, PresupuestoParcialController.getPresupuestoParcialByHistoria);

module.exports = router;
