const express = require("express");
const PresupuestoTotalController = require("../controllers/presupuestototal.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     PresupuestoTotal:
 *       type: object
 *       required:
 *         - ACuenta
 *         - Saldo
 *         - EF
 *         - TC
 *         - Fecha
 *         - PacienteId
 *         - IngresoId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Presupuesto Total
 *         ACuenta:
 *           type: number
 *           description: ACuenta del Presupuesto Total
 *         Saldo:
 *           type: number
 *           description: Saldo del Presupuesto Total
 *         EF:
 *           type: string
 *           description: EF del Presupuesto Total
 *         TC:
 *           type: string
 *           description: TC del Presupuesto Total
 *         Fecha:
 *           type: Date
 *           description: Fecha del Presupuesto Total
 *         PacienteId:
 *           type: number
 *           description: Indentificador de la Historia Clinica
 *         IngresoId:
 *           type: number
 *           description: Indentificador del Ingreso
 *       example:
 *         id: "1"
 *         ACuenta: "124214"
 *         Saldo: "10214"
 *         EF: "fdasfsdf"
 *         TC: "fdsfasdf"
 *         Fecha: "2025-11-21"
 *         PacienteId: "1"
 *         IngresoId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: PresupuestoTotal
 *   description: API para gestionar PresupuestoTotal
 */
/**
 * @swagger
 * /economia/presupuestototal:
 *   get:
 *     summary: Obtiene la lista de PresupuestoTotal
 *     tags: [PresupuestoTotal]
 *     responses:
 *       200:
 *         description: Lista de PresupuestoTotal.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/PresupuestoTotal'
 */
router.get("/", verifyToken, PresupuestoTotalController.getPresupuestoTotalAll);
/**
 * @swagger
 * /economia/presupuestototal/{id}:
 *   get:
 *     summary: Obtiene un PresupuestoTotal por ID
 *     tags: [PresupuestoTotal]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PresupuestoTotal
 *     responses:
 *       200:
 *         description: PresupuestoTotal encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PresupuestoTotal'
 *       404:
 *         description: PresupuestoTotal no encontrado.
 */
router.get("/:id", verifyToken, PresupuestoTotalController.getPresupuestoTotalById);
/**
 * @swagger
 * /economia/presupuestototal/create:
 *   post:
 *     summary: Crea un nuevo PresupuestoTotal
 *     tags: [PresupuestoTotal]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PresupuestoTotal'
 *     responses:
 *       201:
 *         description: PresupuestoTotal creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PresupuestoTotal'
 */
router.post("/create", verifyToken, PresupuestoTotalController.createPresupuestoTotal);
/**
 * @swagger
 * /economia/presupuestototal/update/{id}:
 *   put:
 *     summary: Actualiza un PresupuestoTotal existente
 *     tags: [PresupuestoTotal]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PresupuestoTotal a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/PresupuestoTotal'
 *     responses:
 *       200:
 *         description: PresupuestoTotal actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PresupuestoTotal'
 *       404:
 *         description: PresupuestoTotal no encontrado.
 */
router.put("/update/:id", verifyToken, PresupuestoTotalController.updatePresupuestoTotal);
/**
 * @swagger
 * /economia/presupuestototal/{id}:
 *   delete:
 *     summary: Elimina un PresupuestoTotal por ID
 *     tags: [PresupuestoTotal]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del PresupuestoTotal a eliminar
 *     responses:
 *       200:
 *         description: PresupuestoTotal eliminado exitosamente.
 *       404:
 *         description: PresupuestoTotal no encontrado.
 */
router.delete("/:id", verifyToken, PresupuestoTotalController.deletePresupuestoTotal);
/**
 * @swagger
 * /economia/presupuestototal/filtrar/historia/{historia}:
 *   get:
 *     summary: Obtiene un PresupuestoTotal por usuario
 *     tags: [PresupuestoTotal]
 *     parameters:
 *       - in: path
 *         name: historia
 *         schema:
 *           type: number
 *         required: true
 *         description: historia del PresupuestoTotal
 *     responses:
 *       200:
 *         description: PresupuestoTotal encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/PresupuestoTotal'
 *       404:
 *         description: PresupuestoTotal no encontrado.
 */
router.get("/filtrar/historia/:historia", verifyToken, PresupuestoTotalController.getPresupuestoTotalByHistoria);

module.exports = router;
