const express = require("express");
const ProfesionController = require("../controllers/profesion.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Profesion:
 *       type: object
 *       required:
 *         - Nombre
 *         - Descripcion
 *         - Imagen
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la profesion
 *         Nombre:
 *           type: string
 *           description: Nombre de la Profesion
 *         Descripcion:
 *           type: string
 *           description: Descripcion de la Profesion
 *         Imagen:
 *           type: string
 *           description: Imagen de la Profesion
 *       example:
 *         id: "1"
 *         Nombre: "Odontologo"
 *         Descripcion: "Cuando hablamos de odontología, nos referimos al área de la medicina o la salud que se encarga del tratamiento y prevención de alteraciones o enfermedades que se dan en la boca. Es decir, abarcamos aquí tanto las patologías que afectan a los dientes como las que se dan en las encías o incluso en zonas como la lengua o el paladar."
 *         Imagen: "https://www.bostondentalgroup.com/wp-content/uploads/2015/09/Most-Important-Reasons-to-Visit-the-Dentist.jpg"
 */
/**
 * @swagger
 * tags:
 *   name: Profesion
 *   description: API para gestionar Profesiones
 */
/**
 * @swagger
 * /pilares/profesion:
 *   get:
 *     summary: Obtiene la lista de Profesiones
 *     tags: [Profesion]
 *     responses:
 *       200:
 *         description: Lista de Profesiones.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Profesion'
 */
router.get(
    "/",
    ProfesionController.getProfesionAll
);
/**
 * @swagger
 * /pilares/profesion/{id}:
 *   get:
 *     summary: Obtiene un Profesion por ID
 *     tags: [Profesion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Profesion
 *     responses:
 *       200:
 *         description: Profesion encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Profesion'
 *       404:
 *         description: Profesion no encontrado.
 */
router.get(
    "/:id",
    ProfesionController.getProfesionById
);
/**
 * @swagger
 * /pilares/profesion/create:
 *   post:
 *     summary: Crea un nuevo Profesion
 *     tags: [Profesion]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Profesion'
 *     responses:
 *       201:
 *         description: Profesion creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Profesion'
 */
router.post(
    "/create",
    ProfesionController.createProfesion
);
/**
 * @swagger
 * /pilares/profesion/update/{id}:
 *   put:
 *     summary: Actualiza un Profesion existente
 *     tags: [Profesion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Profesion a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Profesion'
 *     responses:
 *       200:
 *         description: Profesion actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Profesion'
 *       404:
 *         description: Profesion no encontrado.
 */
router.put(
    "/update/:id",
    ProfesionController.updateProfesion
);
/**
 * @swagger
 * /pilares/profesion/{id}:
 *   delete:
 *     summary: Elimina un Profesion por ID
 *     tags: [Profesion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Profesion a eliminar
 *     responses:
 *       200:
 *         description: Profesion eliminado exitosamente.
 *       404:
 *         description: Profesion no encontrado.
 */
router.delete(
    "/:id",
    ProfesionController.deleteProfesion
);

module.exports = router;
