const express = require("express");
const { verifyToken } = require('../middlewares/auth.middleware');
const ProveedorController = require("../controllers/proveedor.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Proveedor:
 *       type: object
 *       required:
 *         - Nombre
 *         - RUC
 *         - SedeId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único de la Proveedor
 *         Nombre:
 *           type: string
 *           description: Nombre del Proveedor
 *         RUC:
 *           type: string
 *           description: RUC del Proveedor
 *         SedeId:
 *           type: id
 *           description: A que sede pertenece el Proveedor
 *       example:
 *         id: 1
 *         Nombre: "Empresa Proveedora S.A."
 *         RUC: "54576586785642"
 *         SedeId: 1

 */
/**
 * @swagger
 * tags:
 *   name: Proveedor
 *   description: API para gestionar Proveedors
 */
/**
 * @swagger
 * /incidencias/Proveedor:
 *   get:
 *     summary: Obtiene la lista de Proveedors
 *     tags: [Proveedor]
 *     responses:
 *       200:
 *         description: Lista de Proveedors.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Proveedor'
 */
router.get("/", verifyToken, ProveedorController.getProveedorAll);
/**
 * @swagger
 * /incidencias/Proveedor/incidencia/{incidencia}:
 *   get:
 *     summary: Obtiene Proveedors asociadas a una Incidencia
 *     tags: [Proveedor]
 *     parameters:
 *       - in: path
 *         name: incidencia
 *         schema:
 *           type: number
 *         required: true
 *         description: ID de la Incidencia
 *     responses:
 *       200:
 *         description: Proveedors encontradas de la Incidencia asociada.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Proveedor'
 *       404:
 *         description: Proveedors no encontrado por la Incidencia Escogida.
 */
router.get("/sede/:sede", verifyToken, ProveedorController.getProveedorBySede);
/**
 * @swagger
 * /incidencias/Proveedor/{id}:
 *   get:
 *     summary: Obtiene un Proveedor por ID
 *     tags: [Proveedor]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Proveedor
 *     responses:
 *       200:
 *         description: Proveedor encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Proveedor'
 *       404:
 *         description: Proveedor no encontrado.
 */
router.get("/:id", verifyToken, ProveedorController.getProveedorById);
/**
 * @swagger
 * /incidencias/Proveedor/create:
 *   post:
 *     summary: Crea un nuevo Proveedor
 *     tags: [Proveedor]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Proveedor'
 *     responses:
 *       201:
 *         description: Proveedor creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Proveedor'
 */
router.post("/create", verifyToken, ProveedorController.createProveedor);
/**
 * @swagger
 * /incidencias/Proveedor/update/{id}:
 *   put:
 *     summary: Actualiza un Proveedor existente
 *     tags: [Proveedor]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Proveedor a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Proveedor'
 *     responses:
 *       200:
 *         description: Proveedor actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Proveedor'
 *       404:
 *         description: Proveedor no encontrado.
 */
router.put("/update/:id", verifyToken, ProveedorController.updateProveedor);
/**
 * @swagger
 * /incidencias/Proveedor/{id}:
 *   delete:
 *     summary: Elimina un Proveedor por ID
 *     tags: [Proveedor]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Proveedor a eliminar
 *     responses:
 *       200:
 *         description: Proveedor eliminado exitosamente.
 *       404:
 *         description: Proveedor no encontrado.
 */
router.delete("/:id", verifyToken, ProveedorController.deleteProveedor);

module.exports = router;
