const express = require("express");
const RecetaController = require("../controllers/receta.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Receta:
 *       type: object
 *       required:
 *         - FechaReceta
 *         - Odontologo
 *         - Paciente
 *         - AgendamientoId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Receta
 *         FechaReceta:
 *           type: Date
 *           description: Fecha de la Receta
 *         Odontologo:
 *           type: string
 *           description: Odontologo que realiza la Receta
 *         Paciente:
 *           type: string
 *           description: Paciente a quien se otorga la Receta
 *         AgendamientoId:
 *           type: number
 *           description: Identificador del Agendamiento
 *         PacienteId:
 *           type: number
 *           description: Identificador de la Historia Clinica
 *       example:
 *         id: "1"
 *         FechaReceta: "2025-11-12"
 *         Odontologo: "fdasfsdf sad "
 *         Paciente: "dsafasdfaasdf "
 *         AgendamientoId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Receta
 *   description: API para gestionar Receta
 */
/**
 * @swagger
 * /paciente/receta:
 *   get:
 *     summary: Obtiene la lista de Receta
 *     tags: [Receta]
 *     responses:
 *       200:
 *         description: Lista de Receta.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Receta'
 */
router.get("/", verifyToken, RecetaController.getRecetaAll);
/**
 * @swagger
 * /paciente/receta/{id}:
 *   get:
 *     summary: Obtiene un Receta por ID
 *     tags: [Receta]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Receta
 *     responses:
 *       200:
 *         description: Receta encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Receta'
 *       404:
 *         description: Receta no encontrado.
 */
router.get("/:id", verifyToken, RecetaController.getRecetaById);
/**
 * @swagger
 * /paciente/receta/create:
 *   post:
 *     summary: Crea un nuevo Receta
 *     tags: [Receta]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Receta'
 *     responses:
 *       201:
 *         description: Receta creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Receta'
 */
router.post("/create", verifyToken, RecetaController.createReceta); // Aplicamos validación
/**
 * @swagger
 * /paciente/receta/update/{id}:
 *   put:
 *     summary: Actualiza un Receta existente
 *     tags: [Receta]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Receta a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Receta'
 *     responses:
 *       200:
 *         description: Receta actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Receta'
 *       404:
 *         description: Receta no encontrado.
 */
router.put("/update/:id", verifyToken, RecetaController.updateReceta);
/**
 * @swagger
 * /paciente/receta/{id}:
 *   delete:
 *     summary: Elimina un Receta por ID
 *     tags: [Receta]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Receta a eliminar
 *     responses:
 *       200:
 *         description: Receta eliminado exitosamente.
 *       404:
 *         description: Receta no encontrado.
 */
router.delete("/:id", verifyToken, RecetaController.deleteReceta);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/receta/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Receta por paciente
 *     tags: [Receta]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: Receta encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Receta'
 *       404:
 *         description: Receta no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/paciente/:paciente", verifyToken, RecetaController.getRecetaByPaciente);

module.exports = router;