const express = require("express");
const ResultadoLaboratorioController = require("../controllers/resultadolaboratorio.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     ResultadoLaboratorio:
 *       type: object
 *       required:
 *         - FechaResultado
 *         - RutaInforme
 *         - Observaciones
 *         - OrdenLaboratorioId
 *         - EmpresaLaboratorioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del ResultadoLaboratorio
 *         FechaResultado:
 *           type: string
 *           description: Fecha del Resultado
 *         RutaInforme:
 *           type: string
 *           description: Ruta Informe de los Resultados
 *         Observaciones:
 *           type: string
 *           description: Observaciones 
 *         OrdenLaboratorioId:
 *           type: string
 *           description: Relacion del ResultadoLaboratorio con Orden de Laboratorio
 *         EmpresaLaboratorioId:
 *           type: string
 *           description: Relacion del ResultadoLaboratorio con Empresa Laboratorio
 *       example:
 *         id: "1"
 *         FechaResultado: "2025-12-13"
 *         RutaInforme: "./Informe/Laboratorio/laboratorio.pdf"
 *         Observaciones: "Ninguna"
 *         OrdenLaboratorioId: "1"
 *         EmpresaLaboratorioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: ResultadoLaboratorio
 *   description: API para gestionar ResultadoLaboratorio
 */
/**
 * @swagger
 * /laboratorio/resultado:
 *   get:
 *     summary: Obtiene la lista de ResultadoLaboratorio
 *     tags: [ResultadoLaboratorio]
 *     responses:
 *       200:
 *         description: Lista de ResultadoLaboratorio.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ResultadoLaboratorio'
 */
router.get("/", verifyToken, ResultadoLaboratorioController.getResultadoLaboratorioAll);
/**
 * @swagger
 * /laboratorio/resultado/{id}:
 *   get:
 *     summary: Obtiene un ResultadoLaboratorio por ID
 *     tags: [ResultadoLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ResultadoLaboratorio
 *     responses:
 *       200:
 *         description: ResultadoLaboratorio encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoLaboratorio'
 *       404:
 *         description: ResultadoLaboratorio no encontrado.
 */
router.get("/:id", verifyToken, ResultadoLaboratorioController.getResultadoLaboratorioById);
/**
 * @swagger
 * /laboratorio/resultado/create:
 *   post:
 *     summary: Crea un nuevo ResultadoLaboratorio
 *     tags: [ResultadoLaboratorio]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ResultadoLaboratorio'
 *     responses:
 *       201:
 *         description: ResultadoLaboratorio creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoLaboratorio'
 */
router.post("/create", verifyToken, ResultadoLaboratorioController.createResultadoLaboratorio);
/**
 * @swagger
 * /laboratorio/resultado/update/{id}:
 *   put:
 *     summary: Actualiza un ResultadoLaboratorio existente
 *     tags: [ResultadoLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ResultadoLaboratorio a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ResultadoLaboratorio'
 *     responses:
 *       200:
 *         description: ResultadoLaboratorio actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoLaboratorio'
 *       404:
 *         description: ResultadoLaboratorio no encontrado.
 */
router.put("/update/:id", verifyToken, ResultadoLaboratorioController.updateResultadoLaboratorio);
/**
 * @swagger
 * /laboratorio/resultado/{id}:
 *   delete:
 *     summary: Elimina un ResultadoLaboratorio por ID
 *     tags: [ResultadoLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ResultadoLaboratorio a eliminar
 *     responses:
 *       200:
 *         description: ResultadoLaboratorio eliminado exitosamente.
 *       404:
 *         description: ResultadoLaboratorio no encontrado.
 */
router.delete("/:id", verifyToken, ResultadoLaboratorioController.deleteResultadoLaboratorio);
/**
 * @swagger
 * /laboratorio/resultado/filtro/empresa/{empresa}:
 *   get:
 *     summary: Obtiene un ResultadoLaboratorio por empresa
 *     tags: [ResultadoLaboratorio]
 *     parameters:
 *       - in: path
 *         name: empresa
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del empresa
 *     responses:
 *       200:
 *         description: ResultadoLaboratorioes del empresa encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoLaboratorio'
 *       404:
 *         description: ResultadoLaboratorio no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/empresa/:empresa", verifyToken, ResultadoLaboratorioController.getResultadoLaboratorioByEmpresa);
/**
 * @swagger
 * /laboratorio/resultado/filtro/orden/{orden}:
 *   get:
 *     summary: Obtiene un ResultadoLaboratorio por orden
 *     tags: [ResultadoLaboratorio]
 *     parameters:
 *       - in: path
 *         name: orden
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del orden
 *     responses:
 *       200:
 *         description: ResultadoLaboratorioes del orden encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoLaboratorio'
 *       404:
 *         description: ResultadoLaboratorio no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/orden/:orden", verifyToken, ResultadoLaboratorioController.getResultadoLaboratorioByOrden);

module.exports = router;
