const express = require("express");
const ResultadoRadiografiaController = require("../controllers/resultadoradiografia.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     ResultadoRadiografia:
 *       type: object
 *       required:
 *         - FechaResultado
 *         - RutaInforme
 *         - Observaciones
 *         - OrdenRadiografiaId
 *         - EmpresaRadiografiaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del ResultadoRadiografia
 *         FechaResultado:
 *           type: string
 *           description: Fecha Resultado del ResultadoRadiografia
 *         RutaInforme:
 *           type: string
 *           description: Ruta Informe del ResultadoRadiografia
 *         Observaciones:
 *           type: string
 *           description: Observaciones del ResultadoRadiografia
 *         OrdenRadiografiaId:
 *           type: string
 *           description: OrdenRadiografia del ResultadoRadiografia
 *         EmpresaRadiografiaId:
 *           type: string
 *           description: EmpresaRadiografia del ResultadoRadiografia
 *       example:
 *         id: "1"
 *         FechaResultado: "2025-12-12"
 *         RutaInforme: "./radiografia/resaf.pdf"
 *         Observaciones: "Ninguna"
 *         OrdenRadiografiaId: "1"
 *         EmpresaRadiografiaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: ResultadoRadiografia
 *   description: API para gestionar ResultadoRadiografia
 */
/**
 * @swagger
 * /radiografia/resultado:
 *   get:
 *     summary: Obtiene la lista de ResultadoRadiografia
 *     tags: [ResultadoRadiografia]
 *     responses:
 *       200:
 *         description: Lista de ResultadoRadiografia.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ResultadoRadiografia'
 */
router.get("/", verifyToken, ResultadoRadiografiaController.getResultadoRadiografiaAll);
/**
 * @swagger
 * /radiografia/resultado/{id}:
 *   get:
 *     summary: Obtiene un ResultadoRadiografia por ID
 *     tags: [ResultadoRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ResultadoRadiografia
 *     responses:
 *       200:
 *         description: ResultadoRadiografia encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoRadiografia'
 *       404:
 *         description: ResultadoRadiografia no encontrado.
 */
router.get("/:id", verifyToken, ResultadoRadiografiaController.getResultadoRadiografiaById);
/**
 * @swagger
 * /radiografia/resultado/create:
 *   post:
 *     summary: Crea un nuevo ResultadoRadiografia
 *     tags: [ResultadoRadiografia]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ResultadoRadiografia'
 *     responses:
 *       201:
 *         description: ResultadoRadiografia creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoRadiografia'
 */
router.post("/create", verifyToken, ResultadoRadiografiaController.createResultadoRadiografia);
/**
 * @swagger
 * /radiografia/resultado/update/{id}:
 *   put:
 *     summary: Actualiza un ResultadoRadiografia existente
 *     tags: [ResultadoRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ResultadoRadiografia a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ResultadoRadiografia'
 *     responses:
 *       200:
 *         description: ResultadoRadiografia actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoRadiografia'
 *       404:
 *         description: ResultadoRadiografia no encontrado.
 */
router.put("/update/:id", verifyToken, ResultadoRadiografiaController.updateResultadoRadiografia);
/**
 * @swagger
 * /radiografia/resultado/{id}:
 *   delete:
 *     summary: Elimina un ResultadoRadiografia por ID
 *     tags: [ResultadoRadiografia]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del ResultadoRadiografia a eliminar
 *     responses:
 *       200:
 *         description: ResultadoRadiografia eliminado exitosamente.
 *       404:
 *         description: ResultadoRadiografia no encontrado.
 */
router.delete("/:id", verifyToken, ResultadoRadiografiaController.deleteResultadoRadiografia);
/**
 * @swagger
 * /radiografia/resultado/filtro/empresa/{empresa}:
 *   get:
 *     summary: Obtiene un ResultadoRadiografia por empresa
 *     tags: [ResultadoRadiografia]
 *     parameters:
 *       - in: path
 *         name: empresa
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del empresa
 *     responses:
 *       200:
 *         description: ResultadoRadiografiaes del empresa encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoRadiografia'
 *       404:
 *         description: ResultadoRadiografia no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/empresa/:empresa", verifyToken, ResultadoRadiografiaController.getResultadoRadiografiaByEmpresaRadiografia);
/**
 * @swagger
 * /radiografia/resultado/filtro/orden/{orden}:
 *   get:
 *     summary: Obtiene un ResultadoRadiografia por orden
 *     tags: [ResultadoRadiografia]
 *     parameters:
 *       - in: path
 *         name: orden
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del orden
 *     responses:
 *       200:
 *         description: ResultadoRadiografiaes del orden encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ResultadoRadiografia'
 *       404:
 *         description: ResultadoRadiografia no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/orden/:orden", verifyToken, ResultadoRadiografiaController.getResultadoRadiografiaByOrdenRadiografia);

module.exports = router;
