const express = require("express");
const sedeController = require("../controllers/sede.controller");
const { verifyToken } = require('../middlewares/auth.middleware');

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Sede:
 *       type: object
 *       required:
 *         - Nombre
 *         - TipoSede
 *         - Direccion
 *         - Referencia
 *         - Encargado
 *         - Foto
 *         - EmpresaId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del usuario
 *         Nombre:
 *           type: string
 *           description: Nombre de la Sede
 *         TipoSede:
 *           type: string
 *           description: Tipo de Sede (Principal, sucursal, expancion y etc)
 *         Direccion:
 *           type: string
 *           description: Direccion de la Sede
 *         Referencia:
 *           type: string
 *           description: Referencia de la Sede
 *         Telefono:
 *           type: string
 *           description: Telefono de la Sede
 *         Celular:
 *           type: string
 *           description: Celular de la Sede
 *         Encargado:
 *           type: string
 *           description: Encargado de la Sede
 *         Foto:
 *           type: string
 *           description: Foto de la Sede
 *         EmpresaId:
 *           type: number
 *           description: A que Empresa pertenece
 *       example:
 *         id: "1"
 *         Nombre: 'Pendiente'
 *         TipoSede: "Principal"
 *         Direccion: "Calle Real Nº 716 - 2do Piso El Tambo – Huancayo"
 *         Referencia: "Costado de la DIRESA"
 *         Telefono: "(064)249898"
 *         Celular: "932050033"
 *         Encargado: ""
 *         Foto: ""
 *         EmpresaId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Sede
 *   description: API para gestionar Sedes
 */
/**
 * @swagger
 * /pilares/sede:
 *   get:
 *     summary: Obtiene la lista de Sedes
 *     tags: [Sede]
 *     responses:
 *       200:
 *         description: Lista de Sedes.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Sede'
 */
router.get(
    "/", verifyToken,
    sedeController.getSedeAll
);
/**
 * @swagger
 * /pilares/sede/empresa/{empresa}:
 *   get:
 *     summary: Obtiene Sedes asociadas a una Empresa
 *     tags: [Sede]
 *     parameters:
 *       - in: path
 *         name: empresa
 *         schema:
 *           type: number
 *         required: true
 *         description: ID de la Empresa
 *     responses:
 *       200:
 *         description: Sedes encontradas de la Empresa asociada.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Sede'
 *       404:
 *         description: Sedes no encontrado por la Empresa Escogida.
 */
router.get(
    "/empresa/:empresa", verifyToken,
    sedeController.getSedeByEmpresa
);
/**
 * @swagger
 * /pilares/sede/{id}:
 *   get:
 *     summary: Obtiene un Sede por ID
 *     tags: [Sede]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Sede
 *     responses:
 *       200:
 *         description: Sede encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Sede'
 *       404:
 *         description: Sede no encontrado.
 */
router.get(
    "/:id", verifyToken,
    sedeController.getSedeById
);
/**
 * @swagger
 * /pilares/sede/create:
 *   post:
 *     summary: Crea un nuevo Sede
 *     tags: [Sede]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Sede'
 *     responses:
 *       201:
 *         description: Sede creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Sede'
 */
router.post(
    "/create", verifyToken,
    sedeController.createSede
);
/**
 * @swagger
 * /pilares/sede/update/{id}:
 *   put:
 *     summary: Actualiza un Sede existente
 *     tags: [Sede]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Sede a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Sede'
 *     responses:
 *       200:
 *         description: Sede actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Sede'
 *       404:
 *         description: Sede no encontrado.
 */
router.put(
    "/update/:id", verifyToken,
    sedeController.updateSede
);
/**
 * @swagger
 * /pilares/sede/{id}:
 *   delete:
 *     summary: Elimina un Sede por ID
 *     tags: [Sede]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Sede a eliminar
 *     responses:
 *       200:
 *         description: Sede eliminado exitosamente.
 *       404:
 *         description: Sede no encontrado.
 */
router.delete(
    "/:id",
    sedeController.deleteSede
);

// Rutas de filtrado individual


module.exports = router;
