const express = require("express");
const ServicioController = require("../controllers/servicio.controller");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Servicio:
 *       type: object
 *       required:
 *         - Nombre
 *         - Descripcion
 *         - Foto
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Servicio
 *         Nombre:
 *           type: string
 *           description: Nombre del Servicio
 *         Descripcion:
 *           type: string
 *           description: Descripcion del Servicio
 *         Foto:
 *           type: string
 *           description: Foto del Servicio
 *       example:
 *         id: "1"
 *         Nombre: "Odontopediatría"
 *         Descripcion: "Creemos en la importancia de la salud bucal desde la aparición de los primeros dientes de leche; momento recomendado para realizar la primera visita al dentista para niños. Por ello, a partir de la erupción dental, te invitamos a traer a los más pequeños de la casa a un examen bucodenta. Es importante la revisión temprana y los consejos para unos buenos hábitos de higiene oral que son fundamentales en la Odontología Infantil."
 *         Foto: "https://clinicadentalimagen.pe/panel/servicio/q.jpg"
 */
/**
 * @swagger
 * tags:
 *   name: Servicio
 *   description: API para gestionar Servicio
 */
/**
 * @swagger
 * /pilares/servicio:
 *   get:
 *     summary: Obtiene la lista de Servicio
 *     tags: [Servicio]
 *     responses:
 *       200:
 *         description: Lista de Servicio.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Servicio'
 */
router.get(
    "/",
    ServicioController.getServicioAll
);
/**
 * @swagger
 * /pilares/servicio/{id}:
 *   get:
 *     summary: Obtiene un Servicio por ID
 *     tags: [Servicio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Servicio
 *     responses:
 *       200:
 *         description: Servicio encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Servicio'
 *       404:
 *         description: Servicio no encontrado.
 */
router.get(
    "/:id",
    ServicioController.getServicioById
);
/**
 * @swagger
 * /pilares/servicio/create:
 *   post:
 *     summary: Crea un nuevo Servicio
 *     tags: [Servicio]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Servicio'
 *     responses:
 *       201:
 *         description: Servicio creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Servicio'
 */
router.post(
    "/create",
    ServicioController.createServicio
);
/**
 * @swagger
 * /pilares/servicio/update/{id}:
 *   put:
 *     summary: Actualiza un Servicio existente
 *     tags: [Servicio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Servicio a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Servicio'
 *     responses:
 *       200:
 *         description: Servicio actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Servicio'
 *       404:
 *         description: Servicio no encontrado.
 */
router.put(
    "/update/:id",
    ServicioController.updateServicio
);
/**
 * @swagger
 * /pilares/servicio/{id}:
 *   delete:
 *     summary: Elimina un Servicio por ID
 *     tags: [Servicio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Servicio a eliminar
 *     responses:
 *       200:
 *         description: Servicio eliminado exitosamente.
 *       404:
 *         description: Servicio no encontrado.
 */
router.delete(
    "/:id",
    ServicioController.deleteServicio
);

module.exports = router;