const express = require("express");
const SubItemLaboratorioController = require("../controllers/subitemlaboratorio.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     SubItemLaboratorio:
 *       type: object
 *       required:
 *         - Nombre
 *         - UnidaddeMedida
 *         - ValorReferencial
 *         - ItemLaboratorioId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del SubItemLaboratorio
 *         Nombre:
 *           type: string
 *           description: Nombre del SubItemLaboratorio
 *         UnidaddeMedida:
 *           type: string
 *           description: Unidad de Medida del SubItemLaboratorio
 *         ValorReferencial:
 *           type: string
 *           description: Valor Referencial del SubItemLaboratorio
 *         ItemLaboratorioId:
 *           type: number
 *           description: Nombre del SubItemLaboratorio
 *       example:
 *         id: "1"
 *         Nombre: "Hemoglobina"
 *         UnidaddeMedida: "g/dL"
 *         ValorReferencial: "En hombres adultos, se consideran normales valores aproximados entre 13,8 y 17,2 g/dL. En mujeres adultas, los rangos típicos oscilan entre 12,1 y 15,1 g/dL."
 *         ItemLaboratorioId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: SubItemLaboratorio
 *   description: API para gestionar SubItemLaboratorio
 */
/**
 * @swagger
 * /laboratorio/subitem:
 *   get:
 *     summary: Obtiene la lista de SubItemLaboratorio
 *     tags: [SubItemLaboratorio]
 *     responses:
 *       200:
 *         description: Lista de SubItemLaboratorio.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/SubItemLaboratorio'
 */
router.get("/", verifyToken, SubItemLaboratorioController.getSubItemLaboratorioAll);
/**
 * @swagger
 * /laboratorio/subitem/{id}:
 *   get:
 *     summary: Obtiene un SubItemLaboratorio por ID
 *     tags: [SubItemLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del SubItemLaboratorio
 *     responses:
 *       200:
 *         description: SubItemLaboratorio encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/SubItemLaboratorio'
 *       404:
 *         description: SubItemLaboratorio no encontrado.
 */
router.get("/:id", verifyToken, SubItemLaboratorioController.getSubItemLaboratorioById);
/**
 * @swagger
 * /laboratorio/subitem/create:
 *   post:
 *     summary: Crea un nuevo SubItemLaboratorio
 *     tags: [SubItemLaboratorio]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/SubItemLaboratorio'
 *     responses:
 *       201:
 *         description: SubItemLaboratorio creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/SubItemLaboratorio'
 */
router.post("/create", verifyToken, SubItemLaboratorioController.createSubItemLaboratorio);
/**
 * @swagger
 * /laboratorio/subitem/update/{id}:
 *   put:
 *     summary: Actualiza un SubItemLaboratorio existente
 *     tags: [SubItemLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del SubItemLaboratorio a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/SubItemLaboratorio'
 *     responses:
 *       200:
 *         description: SubItemLaboratorio actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/SubItemLaboratorio'
 *       404:
 *         description: SubItemLaboratorio no encontrado.
 */
router.put("/update/:id", verifyToken, SubItemLaboratorioController.updateSubItemLaboratorio);
/**
 * @swagger
 * /laboratorio/subitem/{id}:
 *   delete:
 *     summary: Elimina un SubItemLaboratorio por ID
 *     tags: [SubItemLaboratorio]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del SubItemLaboratorio a eliminar
 *     responses:
 *       200:
 *         description: SubItemLaboratorio eliminado exitosamente.
 *       404:
 *         description: SubItemLaboratorio no encontrado.
 */
router.delete("/:id", verifyToken, SubItemLaboratorioController.deleteSubItemLaboratorio);
/**
 * @swagger
 * /laboratorio/subitem/filtro/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un SubItemLaboratorio por paciente
 *     tags: [SubItemLaboratorio]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: SubItemLaboratorioes del paciente encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/SubItemLaboratorio'
 *       404:
 *         description: SubItemLaboratorio no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/itemlaboratorio/:itemlaboratorio", verifyToken, SubItemLaboratorioController.getSubItemLaboratorioByItemLaboratorio);

module.exports = router;
