const express = require("express");
const TarifarioController = require("../controllers/tarifario.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Tarifario:
 *       type: object
 *       required:
 *         - NombreIAFA
 *         - PrecioMinimo
 *         - Precio
 *         - RUCIAFA
 *         - CatalogoId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Tarifario
 *         NombreIAFA:
 *           type: string
 *           description: Nombre IAFA del Tarifario
 *         PrecioMinimo:
 *           type: string
 *           description: Precio Minimol Tarifario
 *         Precio:
 *           type: string
 *           description: Precio del Tarifario
 *         RUCIAFA:
 *           type: string
 *           description: RUCIAFA del Tarifario
 *         CatalogoId:
 *           type: string
 *           description: Catalogo del Tarifario
 *       example:
 *         id: "1"
 *         NombreIAFA: "Rimac Seguros"
 *         PrecioMinimo: "30"
 *         Precio: "35"
 *         RUCIAFA: "2045345245236"
 *         CatalogoId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Tarifario
 *   description: API para gestionar Tarifario
 */
/**
 * @swagger
 * /economia/tarifario:
 *   get:
 *     summary: Obtiene la lista de Tarifario
 *     tags: [Tarifario]
 *     responses:
 *       200:
 *         description: Lista de Tarifario.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Tarifario'
 */
router.get("/", verifyToken, TarifarioController.getTarifarioAll);
/**
 * @swagger
 * /economia/tarifario/{id}:
 *   get:
 *     summary: Obtiene un Tarifario por ID
 *     tags: [Tarifario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Tarifario
 *     responses:
 *       200:
 *         description: Tarifario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Tarifario'
 *       404:
 *         description: Tarifario no encontrado.
 */
router.get("/:id", verifyToken, TarifarioController.getTarifarioById);
/**
 * @swagger
 * /economia/tarifario/create:
 *   post:
 *     summary: Crea un nuevo Tarifario
 *     tags: [Tarifario]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Tarifario'
 *     responses:
 *       201:
 *         description: Tarifario creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Tarifario'
 */
router.post("/create", verifyToken, TarifarioController.createTarifario);
/**
 * @swagger
 * /economia/tarifario/update/{id}:
 *   put:
 *     summary: Actualiza un Tarifario existente
 *     tags: [Tarifario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Tarifario a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Tarifario'
 *     responses:
 *       200:
 *         description: Tarifario actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Tarifario'
 *       404:
 *         description: Tarifario no encontrado.
 */
router.put("/update/:id", verifyToken, TarifarioController.updateTarifario);
/**
 * @swagger
 * /economia/tarifario/{id}:
 *   delete:
 *     summary: Elimina un Tarifario por ID
 *     tags: [Tarifario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Tarifario a eliminar
 *     responses:
 *       200:
 *         description: Tarifario eliminado exitosamente.
 *       404:
 *         description: Tarifario no encontrado.
 */
router.delete("/:id", verifyToken, TarifarioController.deleteTarifario);
/**
 * @swagger
 * /economia/tarifario/filtrar/iafa/{iafa}:
 *   get:
 *     summary: Obtiene un Tarifario por iafa
 *     tags: [Tarifario]
 *     parameters:
 *       - in: path
 *         name: iafa
 *         schema:
 *           type: number
 *         required: true
 *         description: iafa del Tarifario
 *     responses:
 *       200:
 *         description: Tarifario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Tarifario'
 *       404:
 *         description: Tarifario no encontrado.
 */
router.get("/filtrar/iafa/:iafa", verifyToken, TarifarioController.filtrarbyIAFA);
/**
 * @swagger
 * /economia/tarifario/filtrar/catalogo/{catalogo}:
 *   get:
 *     summary: Obtiene un Tarifario por usuario
 *     tags: [Tarifario]
 *     parameters:
 *       - in: path
 *         name: catalogo
 *         schema:
 *           type: number
 *         required: true
 *         description: catalogo del Tarifario
 *     responses:
 *       200:
 *         description: Tarifario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Tarifario'
 *       404:
 *         description: Tarifario no encontrado.
 */
router.get("/filtrar/catalogo/:catalogo", verifyToken, TarifarioController.filtrarbyCatalogo);
// router.get("/filtrar/pais/:pais", TarifarioController.filtrarPorPais);
// router.get("/filtrar/departamento/:departamento", TarifarioController.filtrarPorDepartamento);
// router.get("/filtrar/provincia/:provincia", TarifarioController.filtrarPorProvincia);
// router.get("/filtrar/distrito/:distrito", TarifarioController.filtrarPorDistrito);

module.exports = router;
