const express = require("express");
const TratamientoController = require("../controllers/tratamiento.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
// const { validatePerson } = require("../middlewares/validation.middleware");

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Tratamiento:
 *       type: object
 *       required:
 *         - Cantidad
 *         - NombreCatalogo
 *         - PrecioUnitario
 *         - Subtotal
 *         - IAFA
 *         - CatalogoId
 *         - PlanTratamientoId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Tratamiento
 *         Cantidad:
 *           type: number
 *           description: Cantidad del Tratamiento
 *         NombreCatalogo:
 *           type: string
 *           description: Nombre Catalogo del Tratamiento
 *         PrecioUnitario:
 *           type: number
 *           description: Precio Unitario del Tratamiento
 *         Subtotal:
 *           type: number
 *           description: Sub total del Tratamiento
 *         IAFA:
 *           type: string
 *           description: IAFA del Tratamiento
 *         CatalogoId:
 *           type: number
 *           description: Identificador del Catalogo
 *         PlanTratamientoId:
 *           type: number
 *           description: Identificador del Plan Tratamiento
 *       example:
 *         id: "1"
 *         Cantidad: "2"
 *         NombreCatalogo: "fadsfasdf"
 *         PrecioUnitario: "120"
 *         Subtotal: "240"
 *         IAFA: "sadsgsdgda"
 *         CatalogoId: "1"
 *         PlanTratamientoId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Tratamiento
 *   description: API para gestionar Tratamiento
 */
/**
 * @swagger
 * /paciente/tratamiento:
 *   get:
 *     summary: Obtiene la lista de Tratamiento
 *     tags: [Tratamiento]
 *     responses:
 *       200:
 *         description: Lista de Tratamiento.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Tratamiento'
 */
router.get("/", verifyToken, TratamientoController.getTratamientoAll);
/**
 * @swagger
 * /paciente/tratamiento/{id}:
 *   get:
 *     summary: Obtiene un Tratamiento por ID
 *     tags: [Tratamiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Tratamiento
 *     responses:
 *       200:
 *         description: Tratamiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Tratamiento'
 *       404:
 *         description: Tratamiento no encontrado.
 */
router.get("/:id", verifyToken, TratamientoController.getTratamientoById);
/**
 * @swagger
 * /paciente/tratamiento/create:
 *   post:
 *     summary: Crea un nuevo Tratamiento
 *     tags: [Tratamiento]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Tratamiento'
 *     responses:
 *       201:
 *         description: Tratamiento creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Tratamiento'
 */
router.post("/create", verifyToken, TratamientoController.createTratamiento); // Aplicamos validación
/**
 * @swagger
 * /paciente/tratamiento/update/{id}:
 *   put:
 *     summary: Actualiza un Tratamiento existente
 *     tags: [Tratamiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Tratamiento a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Tratamiento'
 *     responses:
 *       200:
 *         description: Tratamiento actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Tratamiento'
 *       404:
 *         description: Tratamiento no encontrado.
 */
router.put("/update/:id", verifyToken, TratamientoController.updateTratamiento);
/**
 * @swagger
 * /paciente/tratamiento/{id}:
 *   delete:
 *     summary: Elimina un Tratamiento por ID
 *     tags: [Tratamiento]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Tratamiento a eliminar
 *     responses:
 *       200:
 *         description: Tratamiento eliminado exitosamente.
 *       404:
 *         description: Tratamiento no encontrado.
 */
router.delete("/:id", verifyToken, TratamientoController.deleteTratamiento);
// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /paciente/tratamiento/filtrar/plan/{plan}:
 *   get:
 *     summary: Obtiene un Tratamiento por plan
 *     tags: [Tratamiento]
 *     parameters:
 *       - in: path
 *         name: plan
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del plan
 *     responses:
 *       200:
 *         description: Tratamiento encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Tratamiento'
 *       404:
 *         description: Tratamiento no encontrado.
 */
// ruta para filtrar las citas del paciente
router.get("/filtrar/plan/:plan", verifyToken, TratamientoController.getTratamientoByPlanTratamiento);

module.exports = router;